import { format, startOfDay, endOfDay, subDays, startOfWeek, endOfWeek, startOfMonth, endOfMonth, eachDayOfInterval, parseISO, isValid } from 'date-fns';
import { MoodResponse, DailyStats } from '../types';

export const formatTime = (date: Date | number): string => {
  const d = typeof date === 'number' ? new Date(date) : date;
  return format(d, 'h:mm a');
};

export const formatDate = (date: Date | number): string => {
  const d = typeof date === 'number' ? new Date(date) : date;
  return format(d, 'MMM d, yyyy');
};

export const formatDateShort = (date: Date | number): string => {
  const d = typeof date === 'number' ? new Date(date) : date;
  return format(d, 'MMM d');
};

export const formatDateTime = (date: Date | number): string => {
  const d = typeof date === 'number' ? new Date(date) : date;
  return format(d, 'MMM d, yyyy h:mm a');
};

export const getDateKey = (date: Date | number): string => {
  const d = typeof date === 'number' ? new Date(date) : date;
  return format(d, 'yyyy-MM-dd');
};

export const parseTimeString = (timeStr: string): { hours: number; minutes: number } => {
  const [hours, minutes] = timeStr.split(':').map(Number);
  return { hours, minutes };
};

export const getNextScheduledTime = (
  timeStr: string,
  days: number[],
  randomWindow?: number
): Date => {
  const now = new Date();
  const { hours, minutes } = parseTimeString(timeStr);

  // Apply random offset if specified
  let offsetMinutes = 0;
  if (randomWindow) {
    offsetMinutes = Math.floor(Math.random() * (randomWindow * 2 + 1)) - randomWindow;
  }

  for (let i = 0; i < 7; i++) {
    const checkDate = new Date(now);
    checkDate.setDate(checkDate.getDate() + i);
    checkDate.setHours(hours, minutes + offsetMinutes, 0, 0);

    const dayOfWeek = checkDate.getDay();
    if (days.includes(dayOfWeek) && checkDate > now) {
      return checkDate;
    }
  }

  // Fallback: return tomorrow at the scheduled time
  const tomorrow = new Date(now);
  tomorrow.setDate(tomorrow.getDate() + 1);
  tomorrow.setHours(hours, minutes + offsetMinutes, 0, 0);
  return tomorrow;
};

export const calculateDailyStats = (responses: MoodResponse[]): DailyStats[] => {
  const grouped: { [key: string]: MoodResponse[] } = {};

  responses.forEach((response) => {
    const dateKey = getDateKey(response.timestamp);
    if (!grouped[dateKey]) {
      grouped[dateKey] = [];
    }
    grouped[dateKey].push(response);
  });

  return Object.entries(grouped)
    .map(([date, dayResponses]) => {
      const total = dayResponses.reduce((sum, r) => sum + r.rating, 0);
      const count = dayResponses.length;
      const avgMax = dayResponses.reduce((sum, r) => sum + r.scaleMax, 0) / count;
      const average = total / count;

      return {
        date,
        averageRating: Math.round(average * 10) / 10,
        responseCount: count,
        normalizedAverage: average / avgMax,
      };
    })
    .sort((a, b) => a.date.localeCompare(b.date));
};

export const getDateRange = (
  period: 'week' | 'month' | 'custom',
  customStart?: Date,
  customEnd?: Date
): { start: Date; end: Date } => {
  const now = new Date();

  switch (period) {
    case 'week':
      return {
        start: startOfWeek(now, { weekStartsOn: 0 }),
        end: endOfWeek(now, { weekStartsOn: 0 }),
      };
    case 'month':
      return {
        start: startOfMonth(now),
        end: endOfMonth(now),
      };
    case 'custom':
      return {
        start: customStart || subDays(now, 7),
        end: customEnd || now,
      };
    default:
      return {
        start: subDays(now, 7),
        end: now,
      };
  }
};

export const getDaysInRange = (start: Date, end: Date): string[] => {
  return eachDayOfInterval({ start, end }).map((d) => getDateKey(d));
};

export const getRandomQuestion = <T>(items: T[]): T | undefined => {
  if (items.length === 0) return undefined;
  return items[Math.floor(Math.random() * items.length)];
};

export const normalizeRating = (rating: number, max: number): number => {
  return rating / max;
};

export const getRatingLabel = (rating: number, max: number): string => {
  const normalized = normalizeRating(rating, max);
  if (normalized <= 0.2) return 'Very Low';
  if (normalized <= 0.4) return 'Low';
  if (normalized <= 0.6) return 'Moderate';
  if (normalized <= 0.8) return 'Good';
  return 'Excellent';
};

export const generateId = (): string => {
  return `${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
};

export const safeParseDate = (dateStr: string): Date | null => {
  try {
    const parsed = parseISO(dateStr);
    return isValid(parsed) ? parsed : null;
  } catch {
    return null;
  }
};

export const debounce = <T extends (...args: any[]) => any>(
  func: T,
  wait: number
): ((...args: Parameters<T>) => void) => {
  let timeout: ReturnType<typeof setTimeout> | null = null;
  return (...args: Parameters<T>) => {
    if (timeout) clearTimeout(timeout);
    timeout = setTimeout(() => func(...args), wait);
  };
};
