// Core data types for the Mood Tracker app

export interface MoodResponse {
  id: string;
  question: string;
  rating: number;
  scaleMax: number;
  timestamp: number;
  context?: string;
  synced: boolean;
}

export interface Question {
  id: string;
  text: string;
  category?: string;
  active: boolean;
}

export interface Schedule {
  id: string;
  time: string; // HH:MM format
  enabled: boolean;
  days: number[]; // 0-6 for Sunday-Saturday
  randomWindow?: number; // minutes +/- from scheduled time
}

export interface AppSettings {
  scaleMin: number;
  scaleMax: number;
  respectDND: boolean;
  cloudSyncEnabled: boolean;
  onboardingComplete: boolean;
  notificationSound: boolean;
}

export type RatingScale = {
  min: number;
  max: number;
  labels?: { [key: number]: string };
};

export interface DailyStats {
  date: string; // YYYY-MM-DD
  averageRating: number;
  responseCount: number;
  normalizedAverage: number; // 0-1 scale for comparison
}

export interface ChartDataPoint {
  x: Date | string;
  y: number;
  label?: string;
}

// Navigation types
export type RootStackParamList = {
  Onboarding: undefined;
  MainTabs: undefined;
  QuestionEditor: { questionId?: string };
  ScheduleEditor: { scheduleId?: string };
};

export type MainTabParamList = {
  Home: undefined;
  History: undefined;
  Calendar: undefined;
  Settings: undefined;
};

// Store types
export interface AppState {
  settings: AppSettings;
  questions: Question[];
  schedules: Schedule[];
  isLoading: boolean;

  // Actions
  setSettings: (settings: Partial<AppSettings>) => void;
  setQuestions: (questions: Question[]) => void;
  addQuestion: (question: Question) => void;
  updateQuestion: (id: string, updates: Partial<Question>) => void;
  deleteQuestion: (id: string) => void;
  setSchedules: (schedules: Schedule[]) => void;
  addSchedule: (schedule: Schedule) => void;
  updateSchedule: (id: string, updates: Partial<Schedule>) => void;
  deleteSchedule: (id: string) => void;
  setLoading: (loading: boolean) => void;
}

// Database row types
export interface MoodResponseRow {
  id: string;
  question: string;
  rating: number;
  scale_max: number;
  timestamp: number;
  context: string | null;
  synced: number;
}

export interface QuestionRow {
  id: string;
  text: string;
  category: string | null;
  active: number;
}

export interface SettingsRow {
  key: string;
  value: string;
}

export interface ScheduleRow {
  id: string;
  time: string;
  enabled: number;
  days: string;
  random_window: number | null;
}
