import { create } from 'zustand';
import { AppState, AppSettings, Question, Schedule } from '../types';
import {
  getSettings,
  saveSettings,
  getQuestions,
  addQuestion as dbAddQuestion,
  updateQuestion as dbUpdateQuestion,
  deleteQuestion as dbDeleteQuestion,
  getSchedules,
  addSchedule as dbAddSchedule,
  updateSchedule as dbUpdateSchedule,
  deleteSchedule as dbDeleteSchedule,
} from '../services/database';
import { DEFAULT_SETTINGS } from '../utils/constants';
import { rescheduleNotifications } from '../services/scheduler';

export const useAppStore = create<AppState>((set, get) => ({
  settings: DEFAULT_SETTINGS,
  questions: [],
  schedules: [],
  isLoading: true,

  setSettings: async (newSettings: Partial<AppSettings>) => {
    const currentSettings = get().settings;
    const updatedSettings = { ...currentSettings, ...newSettings };

    set({ settings: updatedSettings });

    // Persist to database
    await saveSettings(updatedSettings);

    // Reschedule notifications if scale changed
    if (
      newSettings.scaleMax !== undefined &&
      newSettings.scaleMax !== currentSettings.scaleMax
    ) {
      await rescheduleNotifications();
    }
  },

  setQuestions: (questions: Question[]) => {
    set({ questions });
  },

  addQuestion: async (question: Question) => {
    const { questions } = get();
    set({ questions: [...questions, question] });
    await dbAddQuestion(question);
    // Reschedule to include new question
    await rescheduleNotifications();
  },

  updateQuestion: async (id: string, updates: Partial<Question>) => {
    const { questions } = get();
    const updatedQuestions = questions.map((q) =>
      q.id === id ? { ...q, ...updates } : q
    );
    set({ questions: updatedQuestions });
    await dbUpdateQuestion(id, updates);

    // Reschedule if active status changed
    if (updates.active !== undefined) {
      await rescheduleNotifications();
    }
  },

  deleteQuestion: async (id: string) => {
    const { questions } = get();
    const filteredQuestions = questions.filter((q) => q.id !== id);
    set({ questions: filteredQuestions });
    await dbDeleteQuestion(id);
    await rescheduleNotifications();
  },

  setSchedules: (schedules: Schedule[]) => {
    set({ schedules });
  },

  addSchedule: async (schedule: Schedule) => {
    const { schedules } = get();
    set({ schedules: [...schedules, schedule] });
    await dbAddSchedule(schedule);
    await rescheduleNotifications();
  },

  updateSchedule: async (id: string, updates: Partial<Schedule>) => {
    const { schedules } = get();
    const updatedSchedules = schedules.map((s) =>
      s.id === id ? { ...s, ...updates } : s
    );
    set({ schedules: updatedSchedules });
    await dbUpdateSchedule(id, updates);
    await rescheduleNotifications();
  },

  deleteSchedule: async (id: string) => {
    const { schedules } = get();
    const filteredSchedules = schedules.filter((s) => s.id !== id);
    set({ schedules: filteredSchedules });
    await dbDeleteSchedule(id);
    await rescheduleNotifications();
  },

  setLoading: (loading: boolean) => {
    set({ isLoading: loading });
  },
}));

// Initialize store from database
export const initializeStore = async (): Promise<void> => {
  const store = useAppStore.getState();

  try {
    const [settings, questions, schedules] = await Promise.all([
      getSettings(),
      getQuestions(),
      getSchedules(),
    ]);

    if (settings) {
      useAppStore.setState({ settings });
    }

    useAppStore.setState({
      questions,
      schedules,
      isLoading: false,
    });
  } catch (error) {
    console.error('Failed to initialize store:', error);
    useAppStore.setState({ isLoading: false });
  }
};

// Selectors for common data access patterns
export const useSettings = () => useAppStore((state) => state.settings);
export const useQuestions = () => useAppStore((state) => state.questions);
export const useActiveQuestions = () =>
  useAppStore((state) => state.questions.filter((q) => q.active));
export const useSchedules = () => useAppStore((state) => state.schedules);
export const useEnabledSchedules = () =>
  useAppStore((state) => state.schedules.filter((s) => s.enabled));
export const useIsLoading = () => useAppStore((state) => state.isLoading);
export const useOnboardingComplete = () =>
  useAppStore((state) => state.settings.onboardingComplete);
