import { Schedule, Question, AppSettings } from '../types';
import { getEnabledSchedules, getActiveQuestions, getSettings } from './database';
import {
  scheduleNotification,
  cancelAllNotifications,
  getPendingNotifications,
} from './notifications';
import { getNextScheduledTime, getRandomQuestion } from '../utils/helpers';

// Maximum number of notifications to schedule in advance
const MAX_SCHEDULED_NOTIFICATIONS = 50;
// How many days ahead to schedule
const DAYS_AHEAD = 7;

export interface ScheduledItem {
  scheduleId: string;
  questionId: string;
  triggerTime: Date;
  notificationId?: string;
}

// Calculate all upcoming notification times for the next N days
export const calculateUpcomingTimes = (
  schedules: Schedule[],
  daysAhead: number = DAYS_AHEAD
): { scheduleId: string; triggerTime: Date }[] => {
  const upcomingTimes: { scheduleId: string; triggerTime: Date }[] = [];
  const now = new Date();
  const endDate = new Date(now);
  endDate.setDate(endDate.getDate() + daysAhead);

  for (const schedule of schedules) {
    if (!schedule.enabled || schedule.days.length === 0) continue;

    // Generate times for each day in the range
    let currentDate = new Date(now);
    currentDate.setHours(0, 0, 0, 0);

    while (currentDate <= endDate) {
      const dayOfWeek = currentDate.getDay();

      if (schedule.days.includes(dayOfWeek)) {
        const [hours, minutes] = schedule.time.split(':').map(Number);
        const scheduledTime = new Date(currentDate);
        scheduledTime.setHours(hours, minutes, 0, 0);

        // Apply random window if specified
        if (schedule.randomWindow) {
          const offsetMinutes =
            Math.floor(Math.random() * (schedule.randomWindow * 2 + 1)) -
            schedule.randomWindow;
          scheduledTime.setMinutes(scheduledTime.getMinutes() + offsetMinutes);
        }

        // Only include if in the future
        if (scheduledTime > now) {
          upcomingTimes.push({
            scheduleId: schedule.id,
            triggerTime: scheduledTime,
          });
        }
      }

      currentDate.setDate(currentDate.getDate() + 1);
    }
  }

  // Sort by time and limit to max
  return upcomingTimes
    .sort((a, b) => a.triggerTime.getTime() - b.triggerTime.getTime())
    .slice(0, MAX_SCHEDULED_NOTIFICATIONS);
};

// Schedule all upcoming notifications
export const scheduleAllNotifications = async (): Promise<number> => {
  try {
    // Cancel all existing scheduled notifications first
    await cancelAllNotifications();

    // Get enabled schedules, active questions, and settings
    const [schedules, questions, settings] = await Promise.all([
      getEnabledSchedules(),
      getActiveQuestions(),
      getSettings(),
    ]);

    if (
      schedules.length === 0 ||
      questions.length === 0 ||
      !settings
    ) {
      console.log('No schedules, questions, or settings available');
      return 0;
    }

    // Calculate upcoming times
    const upcomingTimes = calculateUpcomingTimes(schedules);

    // Schedule notifications
    let scheduledCount = 0;

    for (const upcoming of upcomingTimes) {
      // Pick a random question for each notification
      const question = getRandomQuestion(questions);
      if (!question) continue;

      try {
        await scheduleNotification(question, settings, upcoming.triggerTime);
        scheduledCount++;
      } catch (error) {
        console.error('Failed to schedule notification:', error);
      }
    }

    console.log(`Scheduled ${scheduledCount} notifications`);
    return scheduledCount;
  } catch (error) {
    console.error('Failed to schedule notifications:', error);
    throw error;
  }
};

// Reschedule notifications (call when settings change)
export const rescheduleNotifications = async (): Promise<number> => {
  return scheduleAllNotifications();
};

// Get next scheduled notification time
export const getNextNotificationTime = async (): Promise<Date | null> => {
  const schedules = await getEnabledSchedules();

  if (schedules.length === 0) return null;

  const upcomingTimes = calculateUpcomingTimes(schedules, 1);

  if (upcomingTimes.length === 0) return null;

  return upcomingTimes[0].triggerTime;
};

// Check how many notifications are scheduled
export const getScheduledNotificationCount = async (): Promise<number> => {
  const ids = await getPendingNotifications();
  return ids.length;
};

// Trigger an immediate mood check notification
export const triggerImmediateMoodCheck = async (): Promise<void> => {
  const [questions, settings] = await Promise.all([
    getActiveQuestions(),
    getSettings(),
  ]);

  if (!questions.length || !settings) {
    throw new Error('No questions or settings available');
  }

  const question = getRandomQuestion(questions);
  if (!question) {
    throw new Error('No question selected');
  }

  // Import dynamically to avoid circular dependency
  const { showMoodCheckNotification } = await import('./notifications');
  await showMoodCheckNotification(question, settings);
};

// Validate schedule configuration
export const validateSchedule = (schedule: Partial<Schedule>): string[] => {
  const errors: string[] = [];

  if (!schedule.time) {
    errors.push('Time is required');
  } else if (!/^\d{2}:\d{2}$/.test(schedule.time)) {
    errors.push('Time must be in HH:MM format');
  }

  if (!schedule.days || schedule.days.length === 0) {
    errors.push('At least one day must be selected');
  }

  if (schedule.randomWindow !== undefined) {
    if (schedule.randomWindow < 0 || schedule.randomWindow > 120) {
      errors.push('Random window must be between 0 and 120 minutes');
    }
  }

  return errors;
};

// Format schedule for display
export const formatScheduleDescription = (schedule: Schedule): string => {
  const dayNames = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
  const days = schedule.days.map((d) => dayNames[d]).join(', ');

  let description = `${schedule.time} on ${days}`;

  if (schedule.randomWindow) {
    description += ` (±${schedule.randomWindow}min)`;
  }

  return description;
};
