import notifee, {
  AndroidImportance,
  AndroidCategory,
  EventType,
  Event,
  AuthorizationStatus,
  TriggerType,
  TimestampTrigger,
  Notification,
  AndroidAction,
} from '@notifee/react-native';
import { Platform } from 'react-native';
import { NOTIFICATION_CHANNEL_ID, NOTIFICATION_CHANNEL_NAME } from '../utils/constants';
import { addMoodResponse } from './database';
import { Question, AppSettings } from '../types';

// Initialize notification channel for Android
export const initializeNotifications = async (): Promise<void> => {
  if (Platform.OS === 'android') {
    await notifee.createChannel({
      id: NOTIFICATION_CHANNEL_ID,
      name: NOTIFICATION_CHANNEL_NAME,
      importance: AndroidImportance.HIGH,
      vibration: true,
      sound: 'default',
    });
  }
};

// Request notification permissions
export const requestNotificationPermission = async (): Promise<boolean> => {
  const settings = await notifee.requestPermission();
  return (
    settings.authorizationStatus === AuthorizationStatus.AUTHORIZED ||
    settings.authorizationStatus === AuthorizationStatus.PROVISIONAL
  );
};

// Check if notifications are permitted
export const checkNotificationPermission = async (): Promise<boolean> => {
  const settings = await notifee.getNotificationSettings();
  return (
    settings.authorizationStatus === AuthorizationStatus.AUTHORIZED ||
    settings.authorizationStatus === AuthorizationStatus.PROVISIONAL
  );
};

// Generate rating action buttons
const generateRatingActions = (scaleMax: number): AndroidAction[] => {
  const actions: AndroidAction[] = [];

  // For scales up to 5, show all options
  // For larger scales, show key points (1, mid, max)
  if (scaleMax <= 5) {
    for (let i = 1; i <= scaleMax; i++) {
      actions.push({
        title: i.toString(),
        pressAction: { id: `rate_${i}` },
      });
    }
  } else {
    // For 1-10 scale, show: 1, 3, 5, 7, 10 (or similar)
    const step = Math.floor(scaleMax / 4);
    const points = [1];
    for (let i = 1; i < 4; i++) {
      points.push(Math.min(i * step, scaleMax));
    }
    points.push(scaleMax);

    // Remove duplicates and sort
    const uniquePoints = [...new Set(points)].sort((a, b) => a - b);

    for (const point of uniquePoints) {
      actions.push({
        title: point.toString(),
        pressAction: { id: `rate_${point}` },
      });
    }
  }

  return actions;
};

// Display a mood check notification
export const showMoodCheckNotification = async (
  question: Question,
  settings: AppSettings
): Promise<string> => {
  const notificationId = `mood_check_${Date.now()}`;

  const notification: Notification = {
    id: notificationId,
    title: 'Mood Check-in',
    body: question.text,
    data: {
      questionId: question.id,
      questionText: question.text,
      scaleMax: settings.scaleMax.toString(),
      timestamp: Date.now().toString(),
    },
    android: {
      channelId: NOTIFICATION_CHANNEL_ID,
      category: AndroidCategory.REMINDER,
      importance: AndroidImportance.HIGH,
      pressAction: {
        id: 'default',
      },
      actions: generateRatingActions(settings.scaleMax),
      smallIcon: 'ic_notification',
      autoCancel: true,
    },
    ios: {
      categoryId: 'mood_check',
      sound: settings.notificationSound ? 'default' : undefined,
    },
  };

  await notifee.displayNotification(notification);
  return notificationId;
};

// Schedule a notification for a specific time
export const scheduleNotification = async (
  question: Question,
  settings: AppSettings,
  triggerDate: Date
): Promise<string> => {
  const notificationId = `scheduled_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

  const trigger: TimestampTrigger = {
    type: TriggerType.TIMESTAMP,
    timestamp: triggerDate.getTime(),
  };

  await notifee.createTriggerNotification(
    {
      id: notificationId,
      title: 'Mood Check-in',
      body: question.text,
      data: {
        questionId: question.id,
        questionText: question.text,
        scaleMax: settings.scaleMax.toString(),
        timestamp: triggerDate.getTime().toString(),
      },
      android: {
        channelId: NOTIFICATION_CHANNEL_ID,
        category: AndroidCategory.REMINDER,
        importance: AndroidImportance.HIGH,
        pressAction: {
          id: 'default',
        },
        actions: generateRatingActions(settings.scaleMax),
        smallIcon: 'ic_notification',
        autoCancel: true,
      },
      ios: {
        categoryId: 'mood_check',
        sound: settings.notificationSound ? 'default' : undefined,
      },
    },
    trigger
  );

  return notificationId;
};

// Cancel a scheduled notification
export const cancelNotification = async (notificationId: string): Promise<void> => {
  await notifee.cancelNotification(notificationId);
};

// Cancel all scheduled notifications
export const cancelAllNotifications = async (): Promise<void> => {
  await notifee.cancelAllNotifications();
};

// Get all pending notification triggers
export const getPendingNotifications = async (): Promise<string[]> => {
  const triggers = await notifee.getTriggerNotificationIds();
  return triggers;
};

// Handle notification events
export const handleNotificationEvent = async (event: Event): Promise<void> => {
  const { type, detail } = event;

  if (type === EventType.ACTION_PRESS && detail.pressAction) {
    const actionId = detail.pressAction.id;

    // Check if it's a rating action
    if (actionId.startsWith('rate_')) {
      const rating = parseInt(actionId.replace('rate_', ''), 10);
      const notification = detail.notification;

      if (notification?.data) {
        const questionText = notification.data.questionText as string;
        const scaleMax = parseInt(notification.data.scaleMax as string, 10);
        const timestamp = parseInt(notification.data.timestamp as string, 10);

        // Save the response to the database
        await addMoodResponse({
          question: questionText,
          rating,
          scaleMax,
          timestamp: timestamp || Date.now(),
          synced: false,
        });

        // Cancel the notification
        if (notification.id) {
          await notifee.cancelNotification(notification.id);
        }
      }
    }
  } else if (type === EventType.PRESS) {
    // User pressed on the notification body (not an action)
    // This will open the app - handled by navigation
  }
};

// Set up foreground event handler
export const setupForegroundHandler = (): (() => void) => {
  return notifee.onForegroundEvent(handleNotificationEvent);
};

// Set up background event handler (call this at app startup)
export const setupBackgroundHandler = (): void => {
  notifee.onBackgroundEvent(handleNotificationEvent);
};

// Set up iOS notification categories for interactive notifications
export const setupIOSCategories = async (): Promise<void> => {
  if (Platform.OS !== 'ios') return;

  await notifee.setNotificationCategories([
    {
      id: 'mood_check',
      actions: [
        { id: 'rate_1', title: '1' },
        { id: 'rate_2', title: '2' },
        { id: 'rate_3', title: '3' },
        { id: 'rate_4', title: '4' },
        { id: 'rate_5', title: '5' },
      ],
    },
  ]);
};

// Check if Do Not Disturb is active (Android only)
export const isDNDActive = async (): Promise<boolean> => {
  // Notifee doesn't directly expose DND status
  // We'll rely on notification settings instead
  const settings = await notifee.getNotificationSettings();

  if (Platform.OS === 'android') {
    // On Android, check if notifications are blocked
    return settings.authorizationStatus === AuthorizationStatus.DENIED;
  }

  return false;
};

// Get badge count (iOS)
export const getBadgeCount = async (): Promise<number> => {
  return await notifee.getBadgeCount();
};

// Set badge count (iOS)
export const setBadgeCount = async (count: number): Promise<void> => {
  await notifee.setBadgeCount(count);
};

// Clear badge (iOS)
export const clearBadge = async (): Promise<void> => {
  await notifee.setBadgeCount(0);
};
