import SQLite, { SQLiteDatabase, Transaction, ResultSet } from 'react-native-sqlite-storage';
import {
  MoodResponse,
  Question,
  Schedule,
  AppSettings,
  MoodResponseRow,
  QuestionRow,
  ScheduleRow,
} from '../types';
import { DEFAULT_SETTINGS, DEFAULT_QUESTIONS, DEFAULT_SCHEDULES } from '../utils/constants';
import { v4 as uuidv4 } from 'uuid';

SQLite.enablePromise(true);

const DATABASE_NAME = 'moodtracker.db';
const DATABASE_VERSION = 1;

let db: SQLiteDatabase | null = null;

export const initDatabase = async (): Promise<void> => {
  try {
    db = await SQLite.openDatabase({
      name: DATABASE_NAME,
      location: 'default',
    });

    await createTables();
    await initializeDefaults();
  } catch (error) {
    console.error('Failed to initialize database:', error);
    throw error;
  }
};

const createTables = async (): Promise<void> => {
  if (!db) throw new Error('Database not initialized');

  const queries = [
    `CREATE TABLE IF NOT EXISTS mood_responses (
      id TEXT PRIMARY KEY,
      question TEXT NOT NULL,
      rating INTEGER NOT NULL,
      scale_max INTEGER NOT NULL,
      timestamp INTEGER NOT NULL,
      context TEXT,
      synced INTEGER DEFAULT 0
    )`,
    `CREATE TABLE IF NOT EXISTS questions (
      id TEXT PRIMARY KEY,
      text TEXT NOT NULL,
      category TEXT,
      active INTEGER DEFAULT 1
    )`,
    `CREATE TABLE IF NOT EXISTS settings (
      key TEXT PRIMARY KEY,
      value TEXT NOT NULL
    )`,
    `CREATE TABLE IF NOT EXISTS schedules (
      id TEXT PRIMARY KEY,
      time TEXT NOT NULL,
      enabled INTEGER DEFAULT 1,
      days TEXT,
      random_window INTEGER
    )`,
    `CREATE INDEX IF NOT EXISTS idx_mood_timestamp ON mood_responses(timestamp)`,
    `CREATE INDEX IF NOT EXISTS idx_mood_synced ON mood_responses(synced)`,
  ];

  for (const query of queries) {
    await db.executeSql(query);
  }
};

const initializeDefaults = async (): Promise<void> => {
  // Check if settings already exist
  const settings = await getSettings();
  if (!settings) {
    await saveSettings(DEFAULT_SETTINGS);
  }

  // Check if questions already exist
  const questions = await getQuestions();
  if (questions.length === 0) {
    for (const question of DEFAULT_QUESTIONS) {
      await addQuestion(question);
    }
  }

  // Check if schedules already exist
  const schedules = await getSchedules();
  if (schedules.length === 0) {
    for (const schedule of DEFAULT_SCHEDULES) {
      await addSchedule(schedule);
    }
  }
};

// Mood Responses
export const addMoodResponse = async (response: Omit<MoodResponse, 'id'>): Promise<MoodResponse> => {
  if (!db) throw new Error('Database not initialized');

  const id = uuidv4();
  const fullResponse: MoodResponse = { ...response, id };

  await db.executeSql(
    `INSERT INTO mood_responses (id, question, rating, scale_max, timestamp, context, synced)
     VALUES (?, ?, ?, ?, ?, ?, ?)`,
    [
      fullResponse.id,
      fullResponse.question,
      fullResponse.rating,
      fullResponse.scaleMax,
      fullResponse.timestamp,
      fullResponse.context || null,
      fullResponse.synced ? 1 : 0,
    ]
  );

  return fullResponse;
};

export const getMoodResponses = async (
  startTime?: number,
  endTime?: number,
  limit?: number
): Promise<MoodResponse[]> => {
  if (!db) throw new Error('Database not initialized');

  let query = 'SELECT * FROM mood_responses';
  const params: (number | string)[] = [];

  if (startTime !== undefined && endTime !== undefined) {
    query += ' WHERE timestamp >= ? AND timestamp <= ?';
    params.push(startTime, endTime);
  } else if (startTime !== undefined) {
    query += ' WHERE timestamp >= ?';
    params.push(startTime);
  } else if (endTime !== undefined) {
    query += ' WHERE timestamp <= ?';
    params.push(endTime);
  }

  query += ' ORDER BY timestamp DESC';

  if (limit !== undefined) {
    query += ' LIMIT ?';
    params.push(limit);
  }

  const [results] = await db.executeSql(query, params);

  const responses: MoodResponse[] = [];
  for (let i = 0; i < results.rows.length; i++) {
    const row: MoodResponseRow = results.rows.item(i);
    responses.push({
      id: row.id,
      question: row.question,
      rating: row.rating,
      scaleMax: row.scale_max,
      timestamp: row.timestamp,
      context: row.context || undefined,
      synced: row.synced === 1,
    });
  }

  return responses;
};

export const getUnsyncedResponses = async (): Promise<MoodResponse[]> => {
  if (!db) throw new Error('Database not initialized');

  const [results] = await db.executeSql(
    'SELECT * FROM mood_responses WHERE synced = 0 ORDER BY timestamp ASC'
  );

  const responses: MoodResponse[] = [];
  for (let i = 0; i < results.rows.length; i++) {
    const row: MoodResponseRow = results.rows.item(i);
    responses.push({
      id: row.id,
      question: row.question,
      rating: row.rating,
      scaleMax: row.scale_max,
      timestamp: row.timestamp,
      context: row.context || undefined,
      synced: false,
    });
  }

  return responses;
};

export const markResponsesSynced = async (ids: string[]): Promise<void> => {
  if (!db) throw new Error('Database not initialized');
  if (ids.length === 0) return;

  const placeholders = ids.map(() => '?').join(',');
  await db.executeSql(
    `UPDATE mood_responses SET synced = 1 WHERE id IN (${placeholders})`,
    ids
  );
};

export const deleteMoodResponse = async (id: string): Promise<void> => {
  if (!db) throw new Error('Database not initialized');
  await db.executeSql('DELETE FROM mood_responses WHERE id = ?', [id]);
};

// Questions
export const getQuestions = async (): Promise<Question[]> => {
  if (!db) throw new Error('Database not initialized');

  const [results] = await db.executeSql('SELECT * FROM questions ORDER BY text');

  const questions: Question[] = [];
  for (let i = 0; i < results.rows.length; i++) {
    const row: QuestionRow = results.rows.item(i);
    questions.push({
      id: row.id,
      text: row.text,
      category: row.category || undefined,
      active: row.active === 1,
    });
  }

  return questions;
};

export const getActiveQuestions = async (): Promise<Question[]> => {
  if (!db) throw new Error('Database not initialized');

  const [results] = await db.executeSql(
    'SELECT * FROM questions WHERE active = 1 ORDER BY text'
  );

  const questions: Question[] = [];
  for (let i = 0; i < results.rows.length; i++) {
    const row: QuestionRow = results.rows.item(i);
    questions.push({
      id: row.id,
      text: row.text,
      category: row.category || undefined,
      active: true,
    });
  }

  return questions;
};

export const addQuestion = async (question: Question): Promise<void> => {
  if (!db) throw new Error('Database not initialized');

  await db.executeSql(
    'INSERT OR REPLACE INTO questions (id, text, category, active) VALUES (?, ?, ?, ?)',
    [question.id, question.text, question.category || null, question.active ? 1 : 0]
  );
};

export const updateQuestion = async (id: string, updates: Partial<Question>): Promise<void> => {
  if (!db) throw new Error('Database not initialized');

  const setClauses: string[] = [];
  const params: (string | number | null)[] = [];

  if (updates.text !== undefined) {
    setClauses.push('text = ?');
    params.push(updates.text);
  }
  if (updates.category !== undefined) {
    setClauses.push('category = ?');
    params.push(updates.category || null);
  }
  if (updates.active !== undefined) {
    setClauses.push('active = ?');
    params.push(updates.active ? 1 : 0);
  }

  if (setClauses.length === 0) return;

  params.push(id);
  await db.executeSql(
    `UPDATE questions SET ${setClauses.join(', ')} WHERE id = ?`,
    params
  );
};

export const deleteQuestion = async (id: string): Promise<void> => {
  if (!db) throw new Error('Database not initialized');
  await db.executeSql('DELETE FROM questions WHERE id = ?', [id]);
};

// Settings
export const getSettings = async (): Promise<AppSettings | null> => {
  if (!db) throw new Error('Database not initialized');

  const [results] = await db.executeSql('SELECT * FROM settings');

  if (results.rows.length === 0) return null;

  const settings: Record<string, string> = {};
  for (let i = 0; i < results.rows.length; i++) {
    const row = results.rows.item(i);
    settings[row.key] = row.value;
  }

  return {
    scaleMin: parseInt(settings.scaleMin || '1', 10),
    scaleMax: parseInt(settings.scaleMax || '5', 10),
    respectDND: settings.respectDND === 'true',
    cloudSyncEnabled: settings.cloudSyncEnabled === 'true',
    onboardingComplete: settings.onboardingComplete === 'true',
    notificationSound: settings.notificationSound !== 'false',
  };
};

export const saveSettings = async (settings: AppSettings): Promise<void> => {
  if (!db) throw new Error('Database not initialized');

  const entries = [
    ['scaleMin', settings.scaleMin.toString()],
    ['scaleMax', settings.scaleMax.toString()],
    ['respectDND', settings.respectDND.toString()],
    ['cloudSyncEnabled', settings.cloudSyncEnabled.toString()],
    ['onboardingComplete', settings.onboardingComplete.toString()],
    ['notificationSound', settings.notificationSound.toString()],
  ];

  for (const [key, value] of entries) {
    await db.executeSql(
      'INSERT OR REPLACE INTO settings (key, value) VALUES (?, ?)',
      [key, value]
    );
  }
};

export const updateSetting = async (key: string, value: string): Promise<void> => {
  if (!db) throw new Error('Database not initialized');
  await db.executeSql(
    'INSERT OR REPLACE INTO settings (key, value) VALUES (?, ?)',
    [key, value]
  );
};

// Schedules
export const getSchedules = async (): Promise<Schedule[]> => {
  if (!db) throw new Error('Database not initialized');

  const [results] = await db.executeSql('SELECT * FROM schedules ORDER BY time');

  const schedules: Schedule[] = [];
  for (let i = 0; i < results.rows.length; i++) {
    const row: ScheduleRow = results.rows.item(i);
    schedules.push({
      id: row.id,
      time: row.time,
      enabled: row.enabled === 1,
      days: JSON.parse(row.days || '[]'),
      randomWindow: row.random_window || undefined,
    });
  }

  return schedules;
};

export const getEnabledSchedules = async (): Promise<Schedule[]> => {
  const schedules = await getSchedules();
  return schedules.filter((s) => s.enabled);
};

export const addSchedule = async (schedule: Schedule): Promise<void> => {
  if (!db) throw new Error('Database not initialized');

  await db.executeSql(
    'INSERT OR REPLACE INTO schedules (id, time, enabled, days, random_window) VALUES (?, ?, ?, ?, ?)',
    [
      schedule.id,
      schedule.time,
      schedule.enabled ? 1 : 0,
      JSON.stringify(schedule.days),
      schedule.randomWindow || null,
    ]
  );
};

export const updateSchedule = async (id: string, updates: Partial<Schedule>): Promise<void> => {
  if (!db) throw new Error('Database not initialized');

  const setClauses: string[] = [];
  const params: (string | number | null)[] = [];

  if (updates.time !== undefined) {
    setClauses.push('time = ?');
    params.push(updates.time);
  }
  if (updates.enabled !== undefined) {
    setClauses.push('enabled = ?');
    params.push(updates.enabled ? 1 : 0);
  }
  if (updates.days !== undefined) {
    setClauses.push('days = ?');
    params.push(JSON.stringify(updates.days));
  }
  if (updates.randomWindow !== undefined) {
    setClauses.push('random_window = ?');
    params.push(updates.randomWindow);
  }

  if (setClauses.length === 0) return;

  params.push(id);
  await db.executeSql(
    `UPDATE schedules SET ${setClauses.join(', ')} WHERE id = ?`,
    params
  );
};

export const deleteSchedule = async (id: string): Promise<void> => {
  if (!db) throw new Error('Database not initialized');
  await db.executeSql('DELETE FROM schedules WHERE id = ?', [id]);
};

// Stats
export const getResponseStats = async (
  startTime: number,
  endTime: number
): Promise<{
  totalResponses: number;
  averageRating: number;
  uniqueDays: number;
}> => {
  if (!db) throw new Error('Database not initialized');

  const [countResult] = await db.executeSql(
    'SELECT COUNT(*) as count FROM mood_responses WHERE timestamp >= ? AND timestamp <= ?',
    [startTime, endTime]
  );

  const [avgResult] = await db.executeSql(
    'SELECT AVG(CAST(rating AS FLOAT) / scale_max) as avg FROM mood_responses WHERE timestamp >= ? AND timestamp <= ?',
    [startTime, endTime]
  );

  const [daysResult] = await db.executeSql(
    `SELECT COUNT(DISTINCT date(timestamp / 1000, 'unixepoch')) as days
     FROM mood_responses
     WHERE timestamp >= ? AND timestamp <= ?`,
    [startTime, endTime]
  );

  return {
    totalResponses: countResult.rows.item(0).count || 0,
    averageRating: avgResult.rows.item(0).avg || 0,
    uniqueDays: daysResult.rows.item(0).days || 0,
  };
};

export const closeDatabase = async (): Promise<void> => {
  if (db) {
    await db.close();
    db = null;
  }
};
