import BackgroundFetch, { BackgroundFetchStatus } from 'react-native-background-fetch';
import { Platform } from 'react-native';
import { scheduleAllNotifications, getScheduledNotificationCount } from './scheduler';
import { initDatabase } from './database';

const BACKGROUND_TASK_ID = 'com.moodtracker.refresh';

// Initialize background fetch
export const initBackgroundFetch = async (): Promise<void> => {
  try {
    const status = await BackgroundFetch.configure(
      {
        minimumFetchInterval: 15, // Minimum 15 minutes (iOS restriction)
        stopOnTerminate: false,
        startOnBoot: true,
        enableHeadless: true,
        requiredNetworkType: BackgroundFetch.NETWORK_TYPE_NONE,
        requiresBatteryNotLow: false,
        requiresCharging: false,
        requiresDeviceIdle: false,
        requiresStorageNotLow: false,
      },
      async (taskId) => {
        console.log('[BackgroundFetch] Event received:', taskId);
        await onBackgroundFetchEvent(taskId);
        BackgroundFetch.finish(taskId);
      },
      (taskId) => {
        console.log('[BackgroundFetch] Task timed out:', taskId);
        BackgroundFetch.finish(taskId);
      }
    );

    console.log('[BackgroundFetch] Configuration status:', status);

    // Schedule a headless task for Android
    if (Platform.OS === 'android') {
      await BackgroundFetch.scheduleTask({
        taskId: BACKGROUND_TASK_ID,
        delay: 60000, // 1 minute
        periodic: true,
        forceAlarmManager: true,
        stopOnTerminate: false,
        enableHeadless: true,
      });
    }
  } catch (error) {
    console.error('[BackgroundFetch] Failed to configure:', error);
  }
};

// Handle background fetch event
const onBackgroundFetchEvent = async (taskId: string): Promise<void> => {
  console.log('[BackgroundFetch] Processing task:', taskId);

  try {
    // Initialize database if needed
    await initDatabase();

    // Check if we need to reschedule notifications
    const scheduledCount = await getScheduledNotificationCount();

    // If we have fewer than 10 notifications scheduled, refresh them
    if (scheduledCount < 10) {
      console.log('[BackgroundFetch] Rescheduling notifications...');
      await scheduleAllNotifications();
    }

    console.log('[BackgroundFetch] Task completed successfully');
  } catch (error) {
    console.error('[BackgroundFetch] Task failed:', error);
  }
};

// Register headless task for Android
export const registerHeadlessTask = (): void => {
  if (Platform.OS === 'android') {
    BackgroundFetch.registerHeadlessTask(async (event) => {
      const taskId = event.taskId;
      console.log('[BackgroundFetch] Headless task:', taskId);

      await onBackgroundFetchEvent(taskId);
      BackgroundFetch.finish(taskId);
    });
  }
};

// Get background fetch status
export const getBackgroundFetchStatus = async (): Promise<{
  status: BackgroundFetchStatus;
  statusText: string;
}> => {
  const status = await BackgroundFetch.status();

  let statusText: string;
  switch (status) {
    case BackgroundFetch.STATUS_RESTRICTED:
      statusText = 'Background fetch is restricted';
      break;
    case BackgroundFetch.STATUS_DENIED:
      statusText = 'Background fetch is denied';
      break;
    case BackgroundFetch.STATUS_AVAILABLE:
      statusText = 'Background fetch is available';
      break;
    default:
      statusText = 'Unknown status';
  }

  return { status, statusText };
};

// Start background fetch manually
export const startBackgroundFetch = async (): Promise<void> => {
  await BackgroundFetch.start();
};

// Stop background fetch
export const stopBackgroundFetch = async (): Promise<void> => {
  await BackgroundFetch.stop();
};

// Trigger a background fetch event (for testing)
export const simulateBackgroundFetch = async (): Promise<void> => {
  if (__DEV__) {
    await onBackgroundFetchEvent('test-task');
  }
};

// Check if background refresh is enabled (iOS)
export const isBackgroundRefreshEnabled = async (): Promise<boolean> => {
  const { status } = await getBackgroundFetchStatus();
  return status === BackgroundFetch.STATUS_AVAILABLE;
};
