import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  Switch,
  Alert,
  TextInput,
  Modal,
} from 'react-native';
import { v4 as uuidv4 } from 'uuid';

import { Question, Schedule } from '../types';
import { COLORS, SCALE_PRESETS, DAYS_OF_WEEK } from '../utils/constants';
import { useSettings } from '../hooks/useSettings';
import { useAppStore } from '../stores/appStore';
import { formatScheduleDescription } from '../services/scheduler';
import { getScheduledNotificationCount, rescheduleNotifications } from '../services/scheduler';
import { getBackgroundFetchStatus } from '../services/background';

const SettingsScreen: React.FC = () => {
  const { settings, setScale, toggleCloudSync, toggleDND, toggleNotificationSound } =
    useSettings();
  const questions = useAppStore((state) => state.questions);
  const schedules = useAppStore((state) => state.schedules);
  const addQuestion = useAppStore((state) => state.addQuestion);
  const updateQuestion = useAppStore((state) => state.updateQuestion);
  const deleteQuestion = useAppStore((state) => state.deleteQuestion);
  const addSchedule = useAppStore((state) => state.addSchedule);
  const updateSchedule = useAppStore((state) => state.updateSchedule);
  const deleteSchedule = useAppStore((state) => state.deleteSchedule);

  const [scheduledCount, setScheduledCount] = useState(0);
  const [backgroundStatus, setBackgroundStatus] = useState('');
  const [questionModalVisible, setQuestionModalVisible] = useState(false);
  const [scheduleModalVisible, setScheduleModalVisible] = useState(false);
  const [editingQuestion, setEditingQuestion] = useState<Question | null>(null);
  const [editingSchedule, setEditingSchedule] = useState<Schedule | null>(null);
  const [newQuestionText, setNewQuestionText] = useState('');
  const [newScheduleTime, setNewScheduleTime] = useState('09:00');
  const [newScheduleDays, setNewScheduleDays] = useState<number[]>([1, 2, 3, 4, 5]);

  useEffect(() => {
    loadStatus();
  }, []);

  const loadStatus = async () => {
    const count = await getScheduledNotificationCount();
    setScheduledCount(count);

    const { statusText } = await getBackgroundFetchStatus();
    setBackgroundStatus(statusText);
  };

  const handleScaleChange = async (preset: (typeof SCALE_PRESETS)[0]) => {
    await setScale(preset.min, preset.max);
    Alert.alert(
      'Scale Updated',
      `Rating scale changed to ${preset.label}. Notifications will be updated.`
    );
  };

  const handleRefreshNotifications = async () => {
    const count = await rescheduleNotifications();
    setScheduledCount(count);
    Alert.alert('Notifications Refreshed', `${count} notifications scheduled.`);
  };

  // Question management
  const openQuestionModal = (question?: Question) => {
    if (question) {
      setEditingQuestion(question);
      setNewQuestionText(question.text);
    } else {
      setEditingQuestion(null);
      setNewQuestionText('');
    }
    setQuestionModalVisible(true);
  };

  const saveQuestion = async () => {
    if (!newQuestionText.trim()) {
      Alert.alert('Error', 'Please enter a question');
      return;
    }

    if (editingQuestion) {
      await updateQuestion(editingQuestion.id, { text: newQuestionText.trim() });
    } else {
      const newQuestion: Question = {
        id: uuidv4(),
        text: newQuestionText.trim(),
        active: true,
      };
      await addQuestion(newQuestion);
    }

    setQuestionModalVisible(false);
    setNewQuestionText('');
    setEditingQuestion(null);
  };

  const handleDeleteQuestion = (question: Question) => {
    Alert.alert(
      'Delete Question',
      `Are you sure you want to delete "${question.text}"?`,
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: () => deleteQuestion(question.id),
        },
      ]
    );
  };

  // Schedule management
  const openScheduleModal = (schedule?: Schedule) => {
    if (schedule) {
      setEditingSchedule(schedule);
      setNewScheduleTime(schedule.time);
      setNewScheduleDays(schedule.days);
    } else {
      setEditingSchedule(null);
      setNewScheduleTime('09:00');
      setNewScheduleDays([1, 2, 3, 4, 5]);
    }
    setScheduleModalVisible(true);
  };

  const saveSchedule = async () => {
    if (newScheduleDays.length === 0) {
      Alert.alert('Error', 'Please select at least one day');
      return;
    }

    if (editingSchedule) {
      await updateSchedule(editingSchedule.id, {
        time: newScheduleTime,
        days: newScheduleDays,
      });
    } else {
      const newSchedule: Schedule = {
        id: uuidv4(),
        time: newScheduleTime,
        enabled: true,
        days: newScheduleDays,
        randomWindow: 15,
      };
      await addSchedule(newSchedule);
    }

    setScheduleModalVisible(false);
    setEditingSchedule(null);
    loadStatus();
  };

  const handleDeleteSchedule = (schedule: Schedule) => {
    Alert.alert(
      'Delete Schedule',
      'Are you sure you want to delete this schedule?',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            await deleteSchedule(schedule.id);
            loadStatus();
          },
        },
      ]
    );
  };

  const toggleDay = (day: number) => {
    setNewScheduleDays((prev) =>
      prev.includes(day) ? prev.filter((d) => d !== day) : [...prev, day].sort()
    );
  };

  const renderSection = (title: string, children: React.ReactNode) => (
    <View style={styles.section}>
      <Text style={styles.sectionTitle}>{title}</Text>
      <View style={styles.sectionContent}>{children}</View>
    </View>
  );

  const renderSettingRow = (
    label: string,
    description: string,
    control: React.ReactNode
  ) => (
    <View style={styles.settingRow}>
      <View style={styles.settingTextContainer}>
        <Text style={styles.settingLabel}>{label}</Text>
        <Text style={styles.settingDescription}>{description}</Text>
      </View>
      {control}
    </View>
  );

  const renderQuestionModal = () => (
    <Modal
      visible={questionModalVisible}
      animationType="slide"
      transparent={true}
      onRequestClose={() => setQuestionModalVisible(false)}
    >
      <View style={styles.modalOverlay}>
        <View style={styles.modalContent}>
          <Text style={styles.modalTitle}>
            {editingQuestion ? 'Edit Question' : 'Add Question'}
          </Text>
          <TextInput
            style={styles.textInput}
            value={newQuestionText}
            onChangeText={setNewQuestionText}
            placeholder="Enter your question..."
            placeholderTextColor={COLORS.textSecondary}
            multiline
            autoFocus
          />
          <View style={styles.modalButtons}>
            <TouchableOpacity
              style={styles.cancelButton}
              onPress={() => setQuestionModalVisible(false)}
            >
              <Text style={styles.cancelButtonText}>Cancel</Text>
            </TouchableOpacity>
            <TouchableOpacity style={styles.saveButton} onPress={saveQuestion}>
              <Text style={styles.saveButtonText}>Save</Text>
            </TouchableOpacity>
          </View>
        </View>
      </View>
    </Modal>
  );

  const renderScheduleModal = () => (
    <Modal
      visible={scheduleModalVisible}
      animationType="slide"
      transparent={true}
      onRequestClose={() => setScheduleModalVisible(false)}
    >
      <View style={styles.modalOverlay}>
        <View style={styles.modalContent}>
          <Text style={styles.modalTitle}>
            {editingSchedule ? 'Edit Schedule' : 'Add Schedule'}
          </Text>

          <Text style={styles.inputLabel}>Time</Text>
          <TextInput
            style={styles.textInput}
            value={newScheduleTime}
            onChangeText={setNewScheduleTime}
            placeholder="HH:MM (24-hour format)"
            placeholderTextColor={COLORS.textSecondary}
          />

          <Text style={styles.inputLabel}>Days</Text>
          <View style={styles.daysContainer}>
            {DAYS_OF_WEEK.map((day) => (
              <TouchableOpacity
                key={day.value}
                style={[
                  styles.dayButton,
                  newScheduleDays.includes(day.value) && styles.dayButtonSelected,
                ]}
                onPress={() => toggleDay(day.value)}
              >
                <Text
                  style={[
                    styles.dayButtonText,
                    newScheduleDays.includes(day.value) &&
                      styles.dayButtonTextSelected,
                  ]}
                >
                  {day.short}
                </Text>
              </TouchableOpacity>
            ))}
          </View>

          <View style={styles.modalButtons}>
            <TouchableOpacity
              style={styles.cancelButton}
              onPress={() => setScheduleModalVisible(false)}
            >
              <Text style={styles.cancelButtonText}>Cancel</Text>
            </TouchableOpacity>
            <TouchableOpacity style={styles.saveButton} onPress={saveSchedule}>
              <Text style={styles.saveButtonText}>Save</Text>
            </TouchableOpacity>
          </View>
        </View>
      </View>
    </Modal>
  );

  return (
    <ScrollView style={styles.container} showsVerticalScrollIndicator={false}>
      {/* Rating Scale */}
      {renderSection(
        'Rating Scale',
        <View style={styles.scaleOptions}>
          {SCALE_PRESETS.map((preset) => (
            <TouchableOpacity
              key={preset.label}
              style={[
                styles.scaleButton,
                settings.scaleMax === preset.max && styles.scaleButtonActive,
              ]}
              onPress={() => handleScaleChange(preset)}
            >
              <Text
                style={[
                  styles.scaleButtonText,
                  settings.scaleMax === preset.max && styles.scaleButtonTextActive,
                ]}
              >
                {preset.label}
              </Text>
            </TouchableOpacity>
          ))}
        </View>
      )}

      {/* Questions */}
      {renderSection(
        'Check-in Questions',
        <>
          {questions.map((question) => (
            <View key={question.id} style={styles.listItem}>
              <View style={styles.listItemContent}>
                <Text style={styles.listItemText} numberOfLines={2}>
                  {question.text}
                </Text>
                <Switch
                  value={question.active}
                  onValueChange={(value) =>
                    updateQuestion(question.id, { active: value })
                  }
                  trackColor={{ false: COLORS.border, true: COLORS.primary + '80' }}
                  thumbColor={question.active ? COLORS.primary : COLORS.textSecondary}
                />
              </View>
              <View style={styles.listItemActions}>
                <TouchableOpacity
                  onPress={() => openQuestionModal(question)}
                  style={styles.actionButton}
                >
                  <Text style={styles.actionButtonText}>Edit</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  onPress={() => handleDeleteQuestion(question)}
                  style={styles.actionButton}
                >
                  <Text style={[styles.actionButtonText, styles.deleteText]}>
                    Delete
                  </Text>
                </TouchableOpacity>
              </View>
            </View>
          ))}
          <TouchableOpacity
            style={styles.addButton}
            onPress={() => openQuestionModal()}
          >
            <Text style={styles.addButtonText}>+ Add Question</Text>
          </TouchableOpacity>
        </>
      )}

      {/* Schedules */}
      {renderSection(
        'Notification Schedules',
        <>
          {schedules.map((schedule) => (
            <View key={schedule.id} style={styles.listItem}>
              <View style={styles.listItemContent}>
                <Text style={styles.listItemText}>
                  {formatScheduleDescription(schedule)}
                </Text>
                <Switch
                  value={schedule.enabled}
                  onValueChange={(value) =>
                    updateSchedule(schedule.id, { enabled: value })
                  }
                  trackColor={{ false: COLORS.border, true: COLORS.primary + '80' }}
                  thumbColor={schedule.enabled ? COLORS.primary : COLORS.textSecondary}
                />
              </View>
              <View style={styles.listItemActions}>
                <TouchableOpacity
                  onPress={() => openScheduleModal(schedule)}
                  style={styles.actionButton}
                >
                  <Text style={styles.actionButtonText}>Edit</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  onPress={() => handleDeleteSchedule(schedule)}
                  style={styles.actionButton}
                >
                  <Text style={[styles.actionButtonText, styles.deleteText]}>
                    Delete
                  </Text>
                </TouchableOpacity>
              </View>
            </View>
          ))}
          <TouchableOpacity
            style={styles.addButton}
            onPress={() => openScheduleModal()}
          >
            <Text style={styles.addButtonText}>+ Add Schedule</Text>
          </TouchableOpacity>
        </>
      )}

      {/* Preferences */}
      {renderSection(
        'Preferences',
        <>
          {renderSettingRow(
            'Notification Sound',
            'Play sound when notification arrives',
            <Switch
              value={settings.notificationSound}
              onValueChange={toggleNotificationSound}
              trackColor={{ false: COLORS.border, true: COLORS.primary + '80' }}
              thumbColor={settings.notificationSound ? COLORS.primary : COLORS.textSecondary}
            />
          )}
          {renderSettingRow(
            'Respect Do Not Disturb',
            'Delay notifications during DND mode',
            <Switch
              value={settings.respectDND}
              onValueChange={toggleDND}
              trackColor={{ false: COLORS.border, true: COLORS.primary + '80' }}
              thumbColor={settings.respectDND ? COLORS.primary : COLORS.textSecondary}
            />
          )}
          {renderSettingRow(
            'Cloud Sync',
            'Backup your data to the cloud',
            <Switch
              value={settings.cloudSyncEnabled}
              onValueChange={toggleCloudSync}
              trackColor={{ false: COLORS.border, true: COLORS.primary + '80' }}
              thumbColor={settings.cloudSyncEnabled ? COLORS.primary : COLORS.textSecondary}
            />
          )}
        </>
      )}

      {/* Status */}
      {renderSection(
        'Status',
        <>
          <View style={styles.statusRow}>
            <Text style={styles.statusLabel}>Scheduled Notifications</Text>
            <Text style={styles.statusValue}>{scheduledCount}</Text>
          </View>
          <View style={styles.statusRow}>
            <Text style={styles.statusLabel}>Background Refresh</Text>
            <Text style={styles.statusValue}>{backgroundStatus}</Text>
          </View>
          <TouchableOpacity
            style={styles.refreshButton}
            onPress={handleRefreshNotifications}
          >
            <Text style={styles.refreshButtonText}>Refresh Notifications</Text>
          </TouchableOpacity>
        </>
      )}

      <View style={styles.footer}>
        <Text style={styles.footerText}>Mood Tracker v1.0.0</Text>
      </View>

      {renderQuestionModal()}
      {renderScheduleModal()}
    </ScrollView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: COLORS.background,
  },
  section: {
    marginTop: 24,
    paddingHorizontal: 16,
  },
  sectionTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: COLORS.textSecondary,
    marginBottom: 12,
    textTransform: 'uppercase',
    letterSpacing: 0.5,
  },
  sectionContent: {
    backgroundColor: COLORS.surface,
    borderRadius: 12,
    overflow: 'hidden',
  },
  settingRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    padding: 16,
    borderBottomWidth: 1,
    borderBottomColor: COLORS.border,
  },
  settingTextContainer: {
    flex: 1,
    marginRight: 16,
  },
  settingLabel: {
    fontSize: 16,
    fontWeight: '500',
    color: COLORS.text,
    marginBottom: 2,
  },
  settingDescription: {
    fontSize: 13,
    color: COLORS.textSecondary,
  },
  scaleOptions: {
    flexDirection: 'row',
    padding: 16,
    gap: 12,
  },
  scaleButton: {
    flex: 1,
    paddingVertical: 12,
    borderRadius: 8,
    borderWidth: 2,
    borderColor: COLORS.border,
    alignItems: 'center',
  },
  scaleButtonActive: {
    borderColor: COLORS.primary,
    backgroundColor: COLORS.primary + '10',
  },
  scaleButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: COLORS.textSecondary,
  },
  scaleButtonTextActive: {
    color: COLORS.primary,
  },
  listItem: {
    padding: 16,
    borderBottomWidth: 1,
    borderBottomColor: COLORS.border,
  },
  listItemContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  listItemText: {
    flex: 1,
    fontSize: 15,
    color: COLORS.text,
    marginRight: 12,
  },
  listItemActions: {
    flexDirection: 'row',
    marginTop: 8,
    gap: 16,
  },
  actionButton: {
    paddingVertical: 4,
  },
  actionButtonText: {
    fontSize: 14,
    color: COLORS.primary,
  },
  deleteText: {
    color: COLORS.error,
  },
  addButton: {
    padding: 16,
    alignItems: 'center',
  },
  addButtonText: {
    fontSize: 15,
    fontWeight: '500',
    color: COLORS.primary,
  },
  statusRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    padding: 16,
    borderBottomWidth: 1,
    borderBottomColor: COLORS.border,
  },
  statusLabel: {
    fontSize: 15,
    color: COLORS.text,
  },
  statusValue: {
    fontSize: 15,
    fontWeight: '500',
    color: COLORS.textSecondary,
  },
  refreshButton: {
    padding: 16,
    alignItems: 'center',
  },
  refreshButtonText: {
    fontSize: 15,
    fontWeight: '500',
    color: COLORS.primary,
  },
  footer: {
    padding: 32,
    alignItems: 'center',
  },
  footerText: {
    fontSize: 13,
    color: COLORS.textSecondary,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    padding: 24,
  },
  modalContent: {
    backgroundColor: COLORS.surface,
    borderRadius: 16,
    padding: 24,
  },
  modalTitle: {
    fontSize: 20,
    fontWeight: '600',
    color: COLORS.text,
    marginBottom: 16,
  },
  inputLabel: {
    fontSize: 14,
    fontWeight: '500',
    color: COLORS.textSecondary,
    marginBottom: 8,
    marginTop: 16,
  },
  textInput: {
    borderWidth: 1,
    borderColor: COLORS.border,
    borderRadius: 8,
    padding: 12,
    fontSize: 16,
    color: COLORS.text,
    minHeight: 48,
  },
  daysContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: 8,
  },
  dayButton: {
    width: 44,
    height: 44,
    borderRadius: 22,
    justifyContent: 'center',
    alignItems: 'center',
    borderWidth: 2,
    borderColor: COLORS.border,
    backgroundColor: COLORS.surface,
  },
  dayButtonSelected: {
    borderColor: COLORS.primary,
    backgroundColor: COLORS.primary,
  },
  dayButtonText: {
    fontSize: 12,
    fontWeight: '600',
    color: COLORS.textSecondary,
  },
  dayButtonTextSelected: {
    color: COLORS.surface,
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    marginTop: 24,
    gap: 12,
  },
  cancelButton: {
    paddingHorizontal: 20,
    paddingVertical: 12,
  },
  cancelButtonText: {
    fontSize: 16,
    color: COLORS.textSecondary,
  },
  saveButton: {
    backgroundColor: COLORS.primary,
    paddingHorizontal: 24,
    paddingVertical: 12,
    borderRadius: 8,
  },
  saveButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: COLORS.surface,
  },
});

export default SettingsScreen;
