import React, { useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  SafeAreaView,
  ScrollView,
  Alert,
} from 'react-native';
import { COLORS, SCALE_PRESETS, DAYS_OF_WEEK } from '../utils/constants';
import { useSettings } from '../hooks/useSettings';
import { useAppStore } from '../stores/appStore';
import { requestNotificationPermission } from '../services/notifications';
import { scheduleAllNotifications } from '../services/scheduler';
import RatingScale from '../components/RatingScale';

type OnboardingStep = 'welcome' | 'permissions' | 'scale' | 'schedule' | 'done';

const OnboardingScreen: React.FC = () => {
  const [step, setStep] = useState<OnboardingStep>('welcome');
  const [selectedScale, setSelectedScale] = useState(SCALE_PRESETS[0]);
  const [selectedDays, setSelectedDays] = useState<number[]>([0, 1, 2, 3, 4, 5, 6]);
  const { setScale, completeOnboarding } = useSettings();
  const schedules = useAppStore((state) => state.schedules);
  const updateSchedule = useAppStore((state) => state.updateSchedule);

  const handleRequestPermissions = async () => {
    const granted = await requestNotificationPermission();
    if (granted) {
      setStep('scale');
    } else {
      Alert.alert(
        'Notifications Required',
        'Please enable notifications to receive mood check-ins. You can enable them in your device settings.',
        [
          { text: 'Continue Anyway', onPress: () => setStep('scale') },
          { text: 'Cancel', style: 'cancel' },
        ]
      );
    }
  };

  const handleScaleSelect = async () => {
    await setScale(selectedScale.min, selectedScale.max);
    setStep('schedule');
  };

  const toggleDay = (day: number) => {
    setSelectedDays((prev) =>
      prev.includes(day) ? prev.filter((d) => d !== day) : [...prev, day]
    );
  };

  const handleScheduleComplete = async () => {
    // Update all schedules with selected days
    for (const schedule of schedules) {
      await updateSchedule(schedule.id, { days: selectedDays });
    }
    setStep('done');
  };

  const handleFinish = async () => {
    await completeOnboarding();
    // Schedule notifications
    await scheduleAllNotifications();
  };

  const renderWelcome = () => (
    <View style={styles.content}>
      <Text style={styles.emoji}>🧠</Text>
      <Text style={styles.title}>Welcome to Mood Tracker</Text>
      <Text style={styles.description}>
        Track your mood and feelings throughout the day with quick, simple
        check-ins delivered right to your notifications.
      </Text>
      <View style={styles.featureList}>
        <Text style={styles.feature}>✓ Quick rating from notifications</Text>
        <Text style={styles.feature}>✓ Flexible scheduling</Text>
        <Text style={styles.feature}>✓ Beautiful visualizations</Text>
        <Text style={styles.feature}>✓ 100% private & local storage</Text>
      </View>
      <TouchableOpacity
        style={styles.primaryButton}
        onPress={() => setStep('permissions')}
      >
        <Text style={styles.primaryButtonText}>Get Started</Text>
      </TouchableOpacity>
    </View>
  );

  const renderPermissions = () => (
    <View style={styles.content}>
      <Text style={styles.emoji}>🔔</Text>
      <Text style={styles.title}>Enable Notifications</Text>
      <Text style={styles.description}>
        We need your permission to send mood check-in notifications. You can
        respond directly from the notification without opening the app.
      </Text>
      <TouchableOpacity
        style={styles.primaryButton}
        onPress={handleRequestPermissions}
      >
        <Text style={styles.primaryButtonText}>Allow Notifications</Text>
      </TouchableOpacity>
      <TouchableOpacity
        style={styles.secondaryButton}
        onPress={() => setStep('scale')}
      >
        <Text style={styles.secondaryButtonText}>Skip for Now</Text>
      </TouchableOpacity>
    </View>
  );

  const renderScale = () => (
    <View style={styles.content}>
      <Text style={styles.emoji}>📊</Text>
      <Text style={styles.title}>Choose Your Scale</Text>
      <Text style={styles.description}>
        Select the rating scale you'd like to use for your mood check-ins.
      </Text>
      <View style={styles.scaleOptions}>
        {SCALE_PRESETS.map((preset) => (
          <TouchableOpacity
            key={preset.label}
            style={[
              styles.scaleOption,
              selectedScale.label === preset.label && styles.scaleOptionSelected,
            ]}
            onPress={() => setSelectedScale(preset)}
          >
            <Text
              style={[
                styles.scaleOptionText,
                selectedScale.label === preset.label &&
                  styles.scaleOptionTextSelected,
              ]}
            >
              {preset.label}
            </Text>
          </TouchableOpacity>
        ))}
      </View>
      <View style={styles.previewContainer}>
        <Text style={styles.previewLabel}>Preview:</Text>
        <RatingScale
          min={selectedScale.min}
          max={selectedScale.max}
          onSelect={() => {}}
          showLabels
        />
      </View>
      <TouchableOpacity
        style={styles.primaryButton}
        onPress={handleScaleSelect}
      >
        <Text style={styles.primaryButtonText}>Continue</Text>
      </TouchableOpacity>
    </View>
  );

  const renderSchedule = () => (
    <View style={styles.content}>
      <Text style={styles.emoji}>📅</Text>
      <Text style={styles.title}>Set Your Schedule</Text>
      <Text style={styles.description}>
        Choose which days you'd like to receive mood check-ins. You can customize
        specific times later in settings.
      </Text>
      <View style={styles.daysContainer}>
        {DAYS_OF_WEEK.map((day) => (
          <TouchableOpacity
            key={day.value}
            style={[
              styles.dayButton,
              selectedDays.includes(day.value) && styles.dayButtonSelected,
            ]}
            onPress={() => toggleDay(day.value)}
          >
            <Text
              style={[
                styles.dayButtonText,
                selectedDays.includes(day.value) && styles.dayButtonTextSelected,
              ]}
            >
              {day.short}
            </Text>
          </TouchableOpacity>
        ))}
      </View>
      <Text style={styles.scheduleInfo}>
        Default schedule: 9 AM, 2 PM, and 8 PM
      </Text>
      <TouchableOpacity
        style={styles.primaryButton}
        onPress={handleScheduleComplete}
      >
        <Text style={styles.primaryButtonText}>Continue</Text>
      </TouchableOpacity>
    </View>
  );

  const renderDone = () => (
    <View style={styles.content}>
      <Text style={styles.emoji}>🎉</Text>
      <Text style={styles.title}>You're All Set!</Text>
      <Text style={styles.description}>
        Your mood tracking is ready to go. You'll receive your first check-in
        notification at the next scheduled time.
      </Text>
      <View style={styles.tipBox}>
        <Text style={styles.tipTitle}>Quick Tip</Text>
        <Text style={styles.tipText}>
          When you receive a notification, tap the rating number directly to log
          your mood without opening the app!
        </Text>
      </View>
      <TouchableOpacity style={styles.primaryButton} onPress={handleFinish}>
        <Text style={styles.primaryButtonText}>Start Tracking</Text>
      </TouchableOpacity>
    </View>
  );

  return (
    <SafeAreaView style={styles.container}>
      <ScrollView
        contentContainerStyle={styles.scrollContent}
        showsVerticalScrollIndicator={false}
      >
        {step === 'welcome' && renderWelcome()}
        {step === 'permissions' && renderPermissions()}
        {step === 'scale' && renderScale()}
        {step === 'schedule' && renderSchedule()}
        {step === 'done' && renderDone()}
      </ScrollView>
      <View style={styles.progressContainer}>
        {['welcome', 'permissions', 'scale', 'schedule', 'done'].map(
          (s, index) => (
            <View
              key={s}
              style={[
                styles.progressDot,
                step === s && styles.progressDotActive,
                ['welcome', 'permissions', 'scale', 'schedule', 'done'].indexOf(
                  step
                ) > index && styles.progressDotComplete,
              ]}
            />
          )
        )}
      </View>
    </SafeAreaView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: COLORS.background,
  },
  scrollContent: {
    flexGrow: 1,
    justifyContent: 'center',
    padding: 24,
  },
  content: {
    alignItems: 'center',
  },
  emoji: {
    fontSize: 64,
    marginBottom: 24,
  },
  title: {
    fontSize: 28,
    fontWeight: '700',
    color: COLORS.text,
    textAlign: 'center',
    marginBottom: 16,
  },
  description: {
    fontSize: 16,
    color: COLORS.textSecondary,
    textAlign: 'center',
    lineHeight: 24,
    marginBottom: 32,
  },
  featureList: {
    alignSelf: 'stretch',
    marginBottom: 32,
  },
  feature: {
    fontSize: 16,
    color: COLORS.text,
    marginBottom: 12,
    paddingLeft: 8,
  },
  primaryButton: {
    backgroundColor: COLORS.primary,
    paddingHorizontal: 32,
    paddingVertical: 16,
    borderRadius: 12,
    minWidth: 200,
    alignItems: 'center',
  },
  primaryButtonText: {
    color: COLORS.surface,
    fontSize: 18,
    fontWeight: '600',
  },
  secondaryButton: {
    marginTop: 16,
    paddingHorizontal: 32,
    paddingVertical: 12,
  },
  secondaryButtonText: {
    color: COLORS.textSecondary,
    fontSize: 16,
  },
  scaleOptions: {
    flexDirection: 'row',
    marginBottom: 24,
  },
  scaleOption: {
    paddingHorizontal: 24,
    paddingVertical: 12,
    borderRadius: 8,
    borderWidth: 2,
    borderColor: COLORS.border,
    marginHorizontal: 8,
  },
  scaleOptionSelected: {
    borderColor: COLORS.primary,
    backgroundColor: COLORS.primary + '10',
  },
  scaleOptionText: {
    fontSize: 16,
    color: COLORS.textSecondary,
    fontWeight: '500',
  },
  scaleOptionTextSelected: {
    color: COLORS.primary,
  },
  previewContainer: {
    alignSelf: 'stretch',
    marginBottom: 32,
    padding: 16,
    backgroundColor: COLORS.surface,
    borderRadius: 12,
  },
  previewLabel: {
    fontSize: 14,
    color: COLORS.textSecondary,
    marginBottom: 12,
  },
  daysContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'center',
    marginBottom: 16,
    gap: 8,
  },
  dayButton: {
    width: 44,
    height: 44,
    borderRadius: 22,
    justifyContent: 'center',
    alignItems: 'center',
    borderWidth: 2,
    borderColor: COLORS.border,
    backgroundColor: COLORS.surface,
  },
  dayButtonSelected: {
    borderColor: COLORS.primary,
    backgroundColor: COLORS.primary,
  },
  dayButtonText: {
    fontSize: 12,
    fontWeight: '600',
    color: COLORS.textSecondary,
  },
  dayButtonTextSelected: {
    color: COLORS.surface,
  },
  scheduleInfo: {
    fontSize: 14,
    color: COLORS.textSecondary,
    marginBottom: 32,
    textAlign: 'center',
  },
  tipBox: {
    backgroundColor: COLORS.primary + '10',
    padding: 16,
    borderRadius: 12,
    marginBottom: 32,
    alignSelf: 'stretch',
  },
  tipTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: COLORS.primary,
    marginBottom: 8,
  },
  tipText: {
    fontSize: 14,
    color: COLORS.text,
    lineHeight: 20,
  },
  progressContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    paddingBottom: 24,
    gap: 8,
  },
  progressDot: {
    width: 8,
    height: 8,
    borderRadius: 4,
    backgroundColor: COLORS.border,
  },
  progressDotActive: {
    backgroundColor: COLORS.primary,
    width: 24,
  },
  progressDotComplete: {
    backgroundColor: COLORS.primary,
  },
});

export default OnboardingScreen;
