import React, { useCallback, useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  RefreshControl,
  TouchableOpacity,
  Dimensions,
} from 'react-native';
import { useFocusEffect } from '@react-navigation/native';

import { COLORS, getMoodColor } from '../utils/constants';
import { useMoodResponses, useRecentStats, useDailyStats } from '../hooks/useMoodData';
import { useSettings } from '../stores/appStore';
import { triggerImmediateMoodCheck } from '../services/scheduler';
import { getNextNotificationTime } from '../services/scheduler';
import { formatTime, formatDateShort } from '../utils/helpers';
import MoodCard from '../components/MoodCard';

const { width: SCREEN_WIDTH } = Dimensions.get('window');

// Simple bar chart component
const SimpleChart: React.FC<{
  data: { date: string; value: number; max: number }[];
}> = ({ data }) => {
  if (data.length === 0) {
    return (
      <View style={chartStyles.noDataContainer}>
        <Text style={chartStyles.noDataText}>
          Complete a few check-ins to see your trend
        </Text>
      </View>
    );
  }

  const maxBars = 7;
  const displayData = data.slice(-maxBars);

  return (
    <View style={chartStyles.container}>
      <View style={chartStyles.barsContainer}>
        {displayData.map((item, index) => {
          const normalizedValue = item.max > 0 ? item.value / item.max : 0;
          const barHeight = Math.max(normalizedValue * 100, 5);
          const color = getMoodColor(item.value, item.max);

          return (
            <View key={index} style={chartStyles.barWrapper}>
              <View style={chartStyles.barBackground}>
                <View
                  style={[
                    chartStyles.bar,
                    { height: `${barHeight}%`, backgroundColor: color },
                  ]}
                />
              </View>
              <Text style={chartStyles.barLabel} numberOfLines={1}>
                {item.date}
              </Text>
            </View>
          );
        })}
      </View>
    </View>
  );
};

const chartStyles = StyleSheet.create({
  container: {
    paddingVertical: 16,
  },
  barsContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    alignItems: 'flex-end',
    height: 120,
  },
  barWrapper: {
    flex: 1,
    alignItems: 'center',
    marginHorizontal: 4,
  },
  barBackground: {
    width: '100%',
    height: 100,
    backgroundColor: COLORS.border + '50',
    borderRadius: 4,
    overflow: 'hidden',
    justifyContent: 'flex-end',
  },
  bar: {
    width: '100%',
    borderRadius: 4,
    minHeight: 4,
  },
  barLabel: {
    marginTop: 8,
    fontSize: 10,
    color: COLORS.textSecondary,
    textAlign: 'center',
  },
  noDataContainer: {
    padding: 32,
    alignItems: 'center',
  },
  noDataText: {
    fontSize: 14,
    color: COLORS.textSecondary,
    textAlign: 'center',
  },
});

const HomeScreen: React.FC = () => {
  const [refreshing, setRefreshing] = useState(false);
  const [nextNotification, setNextNotification] = useState<Date | null>(null);
  const settings = useSettings();

  const { responses, refetch: refetchResponses } = useMoodResponses(5);
  const { totalResponses, averageRating, uniqueDays, refetch: refetchStats } =
    useRecentStats(7);
  const { stats: dailyStats, refetch: refetchDailyStats } = useDailyStats('week');

  const loadNextNotificationTime = async () => {
    const time = await getNextNotificationTime();
    setNextNotification(time);
  };

  useFocusEffect(
    useCallback(() => {
      loadNextNotificationTime();
    }, [])
  );

  const onRefresh = async () => {
    setRefreshing(true);
    await Promise.all([
      refetchResponses(),
      refetchStats(),
      refetchDailyStats(),
      loadNextNotificationTime(),
    ]);
    setRefreshing(false);
  };

  const handleQuickCheckIn = async () => {
    try {
      await triggerImmediateMoodCheck();
    } catch (error) {
      console.error('Failed to trigger check-in:', error);
    }
  };

  // Prepare chart data
  const chartData = dailyStats.map((stat) => ({
    date: formatDateShort(new Date(stat.date)).split(',')[0].slice(0, 6),
    value: stat.averageRating,
    max: settings.scaleMax,
  }));

  const normalizedAverage = averageRating
    ? Math.round((averageRating * 100) / 1) / 100
    : 0;

  const getMoodEmoji = (normalized: number): string => {
    if (normalized <= 0.2) return '😔';
    if (normalized <= 0.4) return '😐';
    if (normalized <= 0.6) return '🙂';
    if (normalized <= 0.8) return '😊';
    return '😄';
  };

  return (
    <ScrollView
      style={styles.container}
      contentContainerStyle={styles.contentContainer}
      refreshControl={
        <RefreshControl
          refreshing={refreshing}
          onRefresh={onRefresh}
          tintColor={COLORS.primary}
        />
      }
    >
      {/* Header Card */}
      <View style={styles.headerCard}>
        <View style={styles.headerTop}>
          <View>
            <Text style={styles.greeting}>Your Week at a Glance</Text>
            <Text style={styles.subGreeting}>
              {totalResponses} check-ins over {uniqueDays} days
            </Text>
          </View>
          <Text style={styles.moodEmoji}>
            {getMoodEmoji(normalizedAverage)}
          </Text>
        </View>

        {nextNotification && (
          <View style={styles.nextNotification}>
            <Text style={styles.nextLabel}>Next check-in:</Text>
            <Text style={styles.nextTime}>
              {formatDateShort(nextNotification)} at {formatTime(nextNotification)}
            </Text>
          </View>
        )}

        <TouchableOpacity
          style={styles.quickCheckButton}
          onPress={handleQuickCheckIn}
        >
          <Text style={styles.quickCheckButtonText}>Quick Check-In Now</Text>
        </TouchableOpacity>
      </View>

      {/* Stats Row */}
      <View style={styles.statsRow}>
        <View style={styles.statCard}>
          <Text style={styles.statValue}>{totalResponses}</Text>
          <Text style={styles.statLabel}>Responses</Text>
        </View>
        <View style={styles.statCard}>
          <Text style={styles.statValue}>
            {normalizedAverage ? `${Math.round(normalizedAverage * 100)}%` : '--'}
          </Text>
          <Text style={styles.statLabel}>Avg Mood</Text>
        </View>
        <View style={styles.statCard}>
          <Text style={styles.statValue}>{uniqueDays}</Text>
          <Text style={styles.statLabel}>Active Days</Text>
        </View>
      </View>

      {/* Weekly Chart */}
      <View style={styles.chartCard}>
        <Text style={styles.sectionTitle}>Weekly Trend</Text>
        <SimpleChart data={chartData} />
      </View>

      {/* Recent Responses */}
      <View style={styles.recentSection}>
        <Text style={styles.sectionTitle}>Recent Check-ins</Text>
        {responses.length > 0 ? (
          responses.map((response) => (
            <MoodCard key={response.id} response={response} />
          ))
        ) : (
          <View style={styles.noDataContainer}>
            <Text style={styles.noDataText}>No check-ins yet</Text>
            <Text style={styles.noDataSubtext}>
              Tap "Quick Check-In Now" to log your first mood
            </Text>
          </View>
        )}
      </View>
    </ScrollView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: COLORS.background,
  },
  contentContainer: {
    paddingBottom: 24,
  },
  headerCard: {
    backgroundColor: COLORS.primary,
    margin: 16,
    padding: 20,
    borderRadius: 16,
  },
  headerTop: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'flex-start',
    marginBottom: 16,
  },
  greeting: {
    fontSize: 22,
    fontWeight: '700',
    color: COLORS.surface,
    marginBottom: 4,
  },
  subGreeting: {
    fontSize: 14,
    color: COLORS.surface + 'CC',
  },
  moodEmoji: {
    fontSize: 40,
  },
  nextNotification: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 16,
  },
  nextLabel: {
    fontSize: 13,
    color: COLORS.surface + 'AA',
    marginRight: 8,
  },
  nextTime: {
    fontSize: 13,
    fontWeight: '600',
    color: COLORS.surface,
  },
  quickCheckButton: {
    backgroundColor: COLORS.surface,
    paddingVertical: 12,
    borderRadius: 8,
    alignItems: 'center',
  },
  quickCheckButtonText: {
    color: COLORS.primary,
    fontWeight: '600',
    fontSize: 15,
  },
  statsRow: {
    flexDirection: 'row',
    paddingHorizontal: 16,
    marginBottom: 16,
    gap: 12,
  },
  statCard: {
    flex: 1,
    backgroundColor: COLORS.surface,
    padding: 16,
    borderRadius: 12,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.05,
    shadowRadius: 4,
    elevation: 2,
  },
  statValue: {
    fontSize: 24,
    fontWeight: '700',
    color: COLORS.text,
    marginBottom: 4,
  },
  statLabel: {
    fontSize: 12,
    color: COLORS.textSecondary,
  },
  chartCard: {
    backgroundColor: COLORS.surface,
    margin: 16,
    marginTop: 0,
    padding: 16,
    borderRadius: 12,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.05,
    shadowRadius: 4,
    elevation: 2,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: COLORS.text,
    marginBottom: 12,
    paddingHorizontal: 16,
  },
  recentSection: {
    marginTop: 8,
  },
  noDataContainer: {
    padding: 32,
    alignItems: 'center',
  },
  noDataText: {
    fontSize: 15,
    color: COLORS.textSecondary,
    textAlign: 'center',
  },
  noDataSubtext: {
    fontSize: 13,
    color: COLORS.textSecondary,
    marginTop: 8,
    textAlign: 'center',
  },
});

export default HomeScreen;
