import React, { useState, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  FlatList,
  RefreshControl,
  TouchableOpacity,
  Alert,
} from 'react-native';
import { useFocusEffect } from '@react-navigation/native';

import { MoodResponse } from '../types';
import { COLORS } from '../utils/constants';
import { useMoodResponses } from '../hooks/useMoodData';
import { deleteMoodResponse } from '../services/database';
import { formatDate } from '../utils/helpers';
import MoodCard from '../components/MoodCard';

type FilterPeriod = 'all' | 'week' | 'month';

const HistoryScreen: React.FC = () => {
  const [refreshing, setRefreshing] = useState(false);
  const [filter, setFilter] = useState<FilterPeriod>('all');

  const getDateRange = () => {
    const now = new Date();
    switch (filter) {
      case 'week':
        const weekAgo = new Date(now);
        weekAgo.setDate(weekAgo.getDate() - 7);
        return { start: weekAgo, end: now };
      case 'month':
        const monthAgo = new Date(now);
        monthAgo.setMonth(monthAgo.getMonth() - 1);
        return { start: monthAgo, end: now };
      default:
        return { start: undefined, end: undefined };
    }
  };

  const { start, end } = getDateRange();
  const { responses, isLoading, refetch } = useMoodResponses(undefined, start, end);

  useFocusEffect(
    useCallback(() => {
      refetch();
    }, [filter])
  );

  const onRefresh = async () => {
    setRefreshing(true);
    await refetch();
    setRefreshing(false);
  };

  const handleDeleteResponse = (response: MoodResponse) => {
    Alert.alert(
      'Delete Entry',
      'Are you sure you want to delete this mood entry? This cannot be undone.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            await deleteMoodResponse(response.id);
            refetch();
          },
        },
      ]
    );
  };

  // Group responses by date
  const groupedResponses = responses.reduce<{
    [key: string]: MoodResponse[];
  }>((groups, response) => {
    const date = formatDate(response.timestamp);
    if (!groups[date]) {
      groups[date] = [];
    }
    groups[date].push(response);
    return groups;
  }, {});

  const sections = Object.entries(groupedResponses).map(([date, items]) => ({
    date,
    data: items,
  }));

  const renderFilterButton = (period: FilterPeriod, label: string) => (
    <TouchableOpacity
      style={[styles.filterButton, filter === period && styles.filterButtonActive]}
      onPress={() => setFilter(period)}
    >
      <Text
        style={[
          styles.filterButtonText,
          filter === period && styles.filterButtonTextActive,
        ]}
      >
        {label}
      </Text>
    </TouchableOpacity>
  );

  const renderItem = ({ item }: { item: MoodResponse }) => (
    <MoodCard
      response={item}
      showDate={false}
      onPress={() => handleDeleteResponse(item)}
    />
  );

  const renderSectionHeader = (date: string) => (
    <View style={styles.sectionHeader}>
      <Text style={styles.sectionHeaderText}>{date}</Text>
    </View>
  );

  const renderEmpty = () => (
    <View style={styles.emptyContainer}>
      <Text style={styles.emptyEmoji}>📝</Text>
      <Text style={styles.emptyTitle}>No entries yet</Text>
      <Text style={styles.emptySubtitle}>
        Your mood check-in history will appear here
      </Text>
    </View>
  );

  return (
    <View style={styles.container}>
      {/* Filters */}
      <View style={styles.filterContainer}>
        {renderFilterButton('all', 'All Time')}
        {renderFilterButton('month', 'This Month')}
        {renderFilterButton('week', 'This Week')}
      </View>

      {/* Summary */}
      <View style={styles.summaryContainer}>
        <Text style={styles.summaryText}>
          {responses.length} {responses.length === 1 ? 'entry' : 'entries'}
        </Text>
      </View>

      {/* List */}
      <FlatList
        data={sections}
        keyExtractor={(item) => item.date}
        renderItem={({ item: section }) => (
          <View>
            {renderSectionHeader(section.date)}
            {section.data.map((response) => (
              <MoodCard
                key={response.id}
                response={response}
                showDate={false}
                onPress={() => handleDeleteResponse(response)}
              />
            ))}
          </View>
        )}
        contentContainerStyle={[
          styles.listContent,
          sections.length === 0 && styles.listContentEmpty,
        ]}
        refreshControl={
          <RefreshControl
            refreshing={refreshing}
            onRefresh={onRefresh}
            tintColor={COLORS.primary}
          />
        }
        ListEmptyComponent={renderEmpty}
        showsVerticalScrollIndicator={false}
      />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: COLORS.background,
  },
  filterContainer: {
    flexDirection: 'row',
    padding: 16,
    gap: 8,
  },
  filterButton: {
    paddingHorizontal: 16,
    paddingVertical: 8,
    borderRadius: 20,
    backgroundColor: COLORS.surface,
    borderWidth: 1,
    borderColor: COLORS.border,
  },
  filterButtonActive: {
    backgroundColor: COLORS.primary,
    borderColor: COLORS.primary,
  },
  filterButtonText: {
    fontSize: 14,
    fontWeight: '500',
    color: COLORS.textSecondary,
  },
  filterButtonTextActive: {
    color: COLORS.surface,
  },
  summaryContainer: {
    paddingHorizontal: 16,
    paddingBottom: 8,
  },
  summaryText: {
    fontSize: 13,
    color: COLORS.textSecondary,
  },
  listContent: {
    paddingBottom: 24,
  },
  listContentEmpty: {
    flex: 1,
  },
  sectionHeader: {
    backgroundColor: COLORS.background,
    paddingHorizontal: 16,
    paddingVertical: 8,
    marginTop: 8,
  },
  sectionHeaderText: {
    fontSize: 14,
    fontWeight: '600',
    color: COLORS.textSecondary,
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 32,
  },
  emptyEmoji: {
    fontSize: 64,
    marginBottom: 16,
  },
  emptyTitle: {
    fontSize: 20,
    fontWeight: '600',
    color: COLORS.text,
    marginBottom: 8,
  },
  emptySubtitle: {
    fontSize: 15,
    color: COLORS.textSecondary,
    textAlign: 'center',
  },
});

export default HistoryScreen;
