import React, { useState, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Modal,
  FlatList,
  ScrollView,
} from 'react-native';
import { useFocusEffect } from '@react-navigation/native';

import { MoodResponse } from '../types';
import { COLORS, DAYS_OF_WEEK } from '../utils/constants';
import { useCalendarData } from '../hooks/useMoodData';
import { formatDate, formatTime } from '../utils/helpers';
import MoodCard from '../components/MoodCard';

const CalendarScreen: React.FC = () => {
  const today = new Date();
  const [currentYear, setCurrentYear] = useState(today.getFullYear());
  const [currentMonth, setCurrentMonth] = useState(today.getMonth());
  const [selectedDate, setSelectedDate] = useState<Date | null>(null);
  const [selectedResponses, setSelectedResponses] = useState<MoodResponse[]>([]);
  const [modalVisible, setModalVisible] = useState(false);

  const { dataByDate, refetch } = useCalendarData(currentYear, currentMonth);

  useFocusEffect(
    useCallback(() => {
      refetch();
    }, [currentYear, currentMonth])
  );

  const goToPreviousMonth = () => {
    if (currentMonth === 0) {
      setCurrentMonth(11);
      setCurrentYear(currentYear - 1);
    } else {
      setCurrentMonth(currentMonth - 1);
    }
  };

  const goToNextMonth = () => {
    if (currentMonth === 11) {
      setCurrentMonth(0);
      setCurrentYear(currentYear + 1);
    } else {
      setCurrentMonth(currentMonth + 1);
    }
  };

  const goToToday = () => {
    setCurrentYear(today.getFullYear());
    setCurrentMonth(today.getMonth());
  };

  const getMonthName = (month: number): string => {
    return new Date(2000, month, 1).toLocaleString('default', { month: 'long' });
  };

  const getDaysInMonth = (year: number, month: number): number => {
    return new Date(year, month + 1, 0).getDate();
  };

  const getFirstDayOfMonth = (year: number, month: number): number => {
    return new Date(year, month, 1).getDay();
  };

  const handleDayPress = (day: number) => {
    const date = new Date(currentYear, currentMonth, day);
    const dateKey = `${currentYear}-${String(currentMonth + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
    const responses = dataByDate.get(dateKey) || [];

    setSelectedDate(date);
    setSelectedResponses(responses);
    setModalVisible(true);
  };

  const renderCalendarHeader = () => (
    <View style={styles.calendarHeader}>
      <TouchableOpacity onPress={goToPreviousMonth} style={styles.navButton}>
        <Text style={styles.navButtonText}>{'<'}</Text>
      </TouchableOpacity>
      <TouchableOpacity onPress={goToToday}>
        <Text style={styles.monthYear}>
          {getMonthName(currentMonth)} {currentYear}
        </Text>
      </TouchableOpacity>
      <TouchableOpacity onPress={goToNextMonth} style={styles.navButton}>
        <Text style={styles.navButtonText}>{'>'}</Text>
      </TouchableOpacity>
    </View>
  );

  const renderWeekDayHeaders = () => (
    <View style={styles.weekDayRow}>
      {DAYS_OF_WEEK.map((day) => (
        <View key={day.value} style={styles.weekDayCell}>
          <Text style={styles.weekDayText}>{day.short}</Text>
        </View>
      ))}
    </View>
  );

  const renderCalendarDays = () => {
    const daysInMonth = getDaysInMonth(currentYear, currentMonth);
    const firstDay = getFirstDayOfMonth(currentYear, currentMonth);
    const days: React.ReactNode[] = [];

    // Empty cells for days before the first day of the month
    for (let i = 0; i < firstDay; i++) {
      days.push(
        <View key={`empty-${i}`} style={styles.dayCell}>
          <Text style={styles.emptyDayText}></Text>
        </View>
      );
    }

    // Days of the month
    for (let day = 1; day <= daysInMonth; day++) {
      const dateKey = `${currentYear}-${String(currentMonth + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
      const responses = dataByDate.get(dateKey) || [];
      const hasData = responses.length > 0;

      const isToday =
        day === today.getDate() &&
        currentMonth === today.getMonth() &&
        currentYear === today.getFullYear();

      // Calculate average mood
      let moodColor = COLORS.border;
      if (hasData) {
        const avgMood =
          responses.reduce((sum, r) => sum + r.rating / r.scaleMax, 0) /
          responses.length;
        const normalizedRating = Math.round(avgMood * 5);
        moodColor = getMoodColor(normalizedRating, 5);
      }

      days.push(
        <TouchableOpacity
          key={day}
          style={[styles.dayCell, isToday && styles.todayCell]}
          onPress={() => handleDayPress(day)}
        >
          <Text style={[styles.dayText, isToday && styles.todayText]}>{day}</Text>
          {hasData && (
            <View style={[styles.moodIndicator, { backgroundColor: moodColor }]}>
              <Text style={styles.moodCount}>{responses.length}</Text>
            </View>
          )}
        </TouchableOpacity>
      );
    }

    // Fill remaining cells
    const totalCells = Math.ceil((firstDay + daysInMonth) / 7) * 7;
    for (let i = firstDay + daysInMonth; i < totalCells; i++) {
      days.push(
        <View key={`empty-end-${i}`} style={styles.dayCell}>
          <Text style={styles.emptyDayText}></Text>
        </View>
      );
    }

    // Group into weeks
    const weeks: React.ReactNode[][] = [];
    for (let i = 0; i < days.length; i += 7) {
      weeks.push(days.slice(i, i + 7));
    }

    return weeks.map((week, index) => (
      <View key={index} style={styles.weekRow}>
        {week}
      </View>
    ));
  };

  const renderLegend = () => (
    <View style={styles.legend}>
      <View style={styles.legendItem}>
        <View style={[styles.legendDot, { backgroundColor: COLORS.moodLow }]} />
        <Text style={styles.legendText}>Low</Text>
      </View>
      <View style={styles.legendItem}>
        <View style={[styles.legendDot, { backgroundColor: COLORS.moodMedium }]} />
        <Text style={styles.legendText}>Medium</Text>
      </View>
      <View style={styles.legendItem}>
        <View style={[styles.legendDot, { backgroundColor: COLORS.moodHigh }]} />
        <Text style={styles.legendText}>High</Text>
      </View>
    </View>
  );

  const renderModal = () => (
    <Modal
      visible={modalVisible}
      animationType="slide"
      transparent={true}
      onRequestClose={() => setModalVisible(false)}
    >
      <View style={styles.modalOverlay}>
        <View style={styles.modalContent}>
          <View style={styles.modalHeader}>
            <Text style={styles.modalTitle}>
              {selectedDate ? formatDate(selectedDate) : ''}
            </Text>
            <TouchableOpacity
              onPress={() => setModalVisible(false)}
              style={styles.closeButton}
            >
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>

          {selectedResponses.length > 0 ? (
            <FlatList
              data={selectedResponses}
              keyExtractor={(item) => item.id}
              renderItem={({ item }) => (
                <MoodCard response={item} showDate={false} />
              )}
              contentContainerStyle={styles.modalList}
              showsVerticalScrollIndicator={false}
            />
          ) : (
            <View style={styles.noDataContainer}>
              <Text style={styles.noDataText}>No check-ins on this day</Text>
            </View>
          )}
        </View>
      </View>
    </Modal>
  );

  return (
    <View style={styles.container}>
      <View style={styles.calendarContainer}>
        {renderCalendarHeader()}
        {renderWeekDayHeaders()}
        <ScrollView showsVerticalScrollIndicator={false}>
          {renderCalendarDays()}
        </ScrollView>
        {renderLegend()}
      </View>
      {renderModal()}
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: COLORS.background,
  },
  calendarContainer: {
    flex: 1,
    backgroundColor: COLORS.surface,
    margin: 16,
    borderRadius: 16,
    padding: 16,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  calendarHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 16,
  },
  navButton: {
    padding: 12,
  },
  navButtonText: {
    fontSize: 20,
    fontWeight: '600',
    color: COLORS.primary,
  },
  monthYear: {
    fontSize: 20,
    fontWeight: '600',
    color: COLORS.text,
  },
  weekDayRow: {
    flexDirection: 'row',
    marginBottom: 8,
  },
  weekDayCell: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: 8,
  },
  weekDayText: {
    fontSize: 12,
    fontWeight: '600',
    color: COLORS.textSecondary,
  },
  weekRow: {
    flexDirection: 'row',
    marginBottom: 4,
  },
  dayCell: {
    flex: 1,
    aspectRatio: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 4,
  },
  todayCell: {
    backgroundColor: COLORS.primary + '20',
    borderRadius: 8,
  },
  dayText: {
    fontSize: 14,
    fontWeight: '500',
    color: COLORS.text,
  },
  todayText: {
    color: COLORS.primary,
    fontWeight: '700',
  },
  emptyDayText: {
    fontSize: 14,
    color: 'transparent',
  },
  moodIndicator: {
    position: 'absolute',
    bottom: 4,
    width: 16,
    height: 16,
    borderRadius: 8,
    justifyContent: 'center',
    alignItems: 'center',
  },
  moodCount: {
    fontSize: 9,
    fontWeight: '700',
    color: COLORS.surface,
  },
  legend: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 16,
    paddingTop: 16,
    borderTopWidth: 1,
    borderTopColor: COLORS.border,
    gap: 24,
  },
  legendItem: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  legendDot: {
    width: 12,
    height: 12,
    borderRadius: 6,
    marginRight: 6,
  },
  legendText: {
    fontSize: 12,
    color: COLORS.textSecondary,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'flex-end',
  },
  modalContent: {
    backgroundColor: COLORS.surface,
    borderTopLeftRadius: 24,
    borderTopRightRadius: 24,
    maxHeight: '70%',
    minHeight: 200,
  },
  modalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: COLORS.border,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: COLORS.text,
  },
  closeButton: {
    paddingHorizontal: 16,
    paddingVertical: 8,
  },
  closeButtonText: {
    fontSize: 15,
    color: COLORS.primary,
    fontWeight: '500',
  },
  modalList: {
    paddingVertical: 8,
  },
  noDataContainer: {
    padding: 32,
    alignItems: 'center',
  },
  noDataText: {
    fontSize: 15,
    color: COLORS.textSecondary,
  },
});

// Helper function for the component
const getMoodColor = (rating: number, max: number): string => {
  const normalized = rating / max;
  if (normalized <= 0.33) return COLORS.moodLow;
  if (normalized <= 0.66) return COLORS.moodMedium;
  return COLORS.moodHigh;
};

export default CalendarScreen;
