import { useCallback } from 'react';
import { useAppStore, useSettings as useSettingsSelector } from '../stores/appStore';
import { AppSettings } from '../types';

export interface UseSettingsResult {
  settings: AppSettings;
  updateSettings: (updates: Partial<AppSettings>) => Promise<void>;
  setScale: (min: number, max: number) => Promise<void>;
  toggleCloudSync: () => Promise<void>;
  toggleDND: () => Promise<void>;
  toggleNotificationSound: () => Promise<void>;
  completeOnboarding: () => Promise<void>;
}

export const useSettings = (): UseSettingsResult => {
  const settings = useSettingsSelector();
  const setSettings = useAppStore((state) => state.setSettings);

  const updateSettings = useCallback(
    async (updates: Partial<AppSettings>) => {
      await setSettings(updates);
    },
    [setSettings]
  );

  const setScale = useCallback(
    async (min: number, max: number) => {
      await setSettings({ scaleMin: min, scaleMax: max });
    },
    [setSettings]
  );

  const toggleCloudSync = useCallback(async () => {
    await setSettings({ cloudSyncEnabled: !settings.cloudSyncEnabled });
  }, [settings.cloudSyncEnabled, setSettings]);

  const toggleDND = useCallback(async () => {
    await setSettings({ respectDND: !settings.respectDND });
  }, [settings.respectDND, setSettings]);

  const toggleNotificationSound = useCallback(async () => {
    await setSettings({ notificationSound: !settings.notificationSound });
  }, [settings.notificationSound, setSettings]);

  const completeOnboarding = useCallback(async () => {
    await setSettings({ onboardingComplete: true });
  }, [setSettings]);

  return {
    settings,
    updateSettings,
    setScale,
    toggleCloudSync,
    toggleDND,
    toggleNotificationSound,
    completeOnboarding,
  };
};
