import { useState, useEffect, useCallback } from 'react';
import { MoodResponse, DailyStats } from '../types';
import { getMoodResponses, getResponseStats } from '../services/database';
import { calculateDailyStats, getDateRange } from '../utils/helpers';
import { startOfDay, endOfDay, subDays } from 'date-fns';

export interface UseMoodDataResult {
  responses: MoodResponse[];
  isLoading: boolean;
  error: Error | null;
  refetch: () => Promise<void>;
}

export const useMoodResponses = (
  limit?: number,
  startDate?: Date,
  endDate?: Date
): UseMoodDataResult => {
  const [responses, setResponses] = useState<MoodResponse[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<Error | null>(null);

  const fetchData = useCallback(async () => {
    setIsLoading(true);
    setError(null);

    try {
      const startTime = startDate ? startOfDay(startDate).getTime() : undefined;
      const endTime = endDate ? endOfDay(endDate).getTime() : undefined;

      const data = await getMoodResponses(startTime, endTime, limit);
      setResponses(data);
    } catch (err) {
      setError(err instanceof Error ? err : new Error('Failed to fetch responses'));
    } finally {
      setIsLoading(false);
    }
  }, [limit, startDate?.getTime(), endDate?.getTime()]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  return { responses, isLoading, error, refetch: fetchData };
};

export interface UseDailyStatsResult {
  stats: DailyStats[];
  isLoading: boolean;
  error: Error | null;
  refetch: () => Promise<void>;
}

export const useDailyStats = (
  period: 'week' | 'month' | 'custom' = 'week',
  customStart?: Date,
  customEnd?: Date
): UseDailyStatsResult => {
  const [stats, setStats] = useState<DailyStats[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<Error | null>(null);

  const fetchData = useCallback(async () => {
    setIsLoading(true);
    setError(null);

    try {
      const { start, end } = getDateRange(period, customStart, customEnd);
      const responses = await getMoodResponses(
        startOfDay(start).getTime(),
        endOfDay(end).getTime()
      );

      const dailyStats = calculateDailyStats(responses);
      setStats(dailyStats);
    } catch (err) {
      setError(err instanceof Error ? err : new Error('Failed to fetch stats'));
    } finally {
      setIsLoading(false);
    }
  }, [period, customStart?.getTime(), customEnd?.getTime()]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  return { stats, isLoading, error, refetch: fetchData };
};

export interface UseRecentStatsResult {
  totalResponses: number;
  averageRating: number;
  uniqueDays: number;
  isLoading: boolean;
  error: Error | null;
  refetch: () => Promise<void>;
}

export const useRecentStats = (days: number = 7): UseRecentStatsResult => {
  const [totalResponses, setTotalResponses] = useState(0);
  const [averageRating, setAverageRating] = useState(0);
  const [uniqueDays, setUniqueDays] = useState(0);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<Error | null>(null);

  const fetchData = useCallback(async () => {
    setIsLoading(true);
    setError(null);

    try {
      const now = new Date();
      const startDate = subDays(now, days);

      const stats = await getResponseStats(
        startOfDay(startDate).getTime(),
        endOfDay(now).getTime()
      );

      setTotalResponses(stats.totalResponses);
      setAverageRating(stats.averageRating);
      setUniqueDays(stats.uniqueDays);
    } catch (err) {
      setError(err instanceof Error ? err : new Error('Failed to fetch stats'));
    } finally {
      setIsLoading(false);
    }
  }, [days]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  return {
    totalResponses,
    averageRating,
    uniqueDays,
    isLoading,
    error,
    refetch: fetchData,
  };
};

export interface UseCalendarDataResult {
  dataByDate: Map<string, MoodResponse[]>;
  isLoading: boolean;
  error: Error | null;
  refetch: () => Promise<void>;
}

export const useCalendarData = (
  year: number,
  month: number
): UseCalendarDataResult => {
  const [dataByDate, setDataByDate] = useState<Map<string, MoodResponse[]>>(
    new Map()
  );
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<Error | null>(null);

  const fetchData = useCallback(async () => {
    setIsLoading(true);
    setError(null);

    try {
      const startDate = new Date(year, month, 1);
      const endDate = new Date(year, month + 1, 0);

      const responses = await getMoodResponses(
        startOfDay(startDate).getTime(),
        endOfDay(endDate).getTime()
      );

      const grouped = new Map<string, MoodResponse[]>();
      responses.forEach((response) => {
        const date = new Date(response.timestamp);
        const key = `${date.getFullYear()}-${String(date.getMonth() + 1).padStart(
          2,
          '0'
        )}-${String(date.getDate()).padStart(2, '0')}`;

        if (!grouped.has(key)) {
          grouped.set(key, []);
        }
        grouped.get(key)!.push(response);
      });

      setDataByDate(grouped);
    } catch (err) {
      setError(err instanceof Error ? err : new Error('Failed to fetch calendar data'));
    } finally {
      setIsLoading(false);
    }
  }, [year, month]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  return { dataByDate, isLoading, error, refetch: fetchData };
};
