import React from 'react';
import {
  View,
  TouchableOpacity,
  Text,
  StyleSheet,
  Dimensions,
} from 'react-native';
import { COLORS, getMoodColor } from '../utils/constants';

interface RatingScaleProps {
  min: number;
  max: number;
  value?: number;
  onSelect: (rating: number) => void;
  disabled?: boolean;
  size?: 'small' | 'medium' | 'large';
  showLabels?: boolean;
  labels?: { [key: number]: string };
}

const { width: SCREEN_WIDTH } = Dimensions.get('window');

export const RatingScale: React.FC<RatingScaleProps> = ({
  min,
  max,
  value,
  onSelect,
  disabled = false,
  size = 'medium',
  showLabels = false,
  labels,
}) => {
  const ratings = Array.from({ length: max - min + 1 }, (_, i) => min + i);

  const buttonSize = {
    small: 32,
    medium: 44,
    large: 56,
  }[size];

  const fontSize = {
    small: 12,
    medium: 16,
    large: 20,
  }[size];

  const getButtonStyle = (rating: number) => {
    const isSelected = value === rating;
    const color = getMoodColor(rating, max);

    return [
      styles.button,
      {
        width: buttonSize,
        height: buttonSize,
        borderRadius: buttonSize / 2,
        backgroundColor: isSelected ? color : COLORS.surface,
        borderColor: isSelected ? color : COLORS.border,
      },
      disabled && styles.buttonDisabled,
    ];
  };

  const getTextStyle = (rating: number) => {
    const isSelected = value === rating;

    return [
      styles.buttonText,
      {
        fontSize,
        color: isSelected ? COLORS.surface : COLORS.text,
      },
    ];
  };

  const getLabel = (rating: number): string | undefined => {
    if (labels && labels[rating]) {
      return labels[rating];
    }
    if (rating === min) return 'Low';
    if (rating === max) return 'High';
    return undefined;
  };

  return (
    <View style={styles.container}>
      <View style={styles.scaleContainer}>
        {ratings.map((rating) => (
          <View key={rating} style={styles.buttonWrapper}>
            <TouchableOpacity
              style={getButtonStyle(rating)}
              onPress={() => !disabled && onSelect(rating)}
              disabled={disabled}
              activeOpacity={0.7}
            >
              <Text style={getTextStyle(rating)}>{rating}</Text>
            </TouchableOpacity>
            {showLabels && (
              <Text style={styles.label} numberOfLines={1}>
                {getLabel(rating) || ''}
              </Text>
            )}
          </View>
        ))}
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    width: '100%',
    paddingVertical: 8,
  },
  scaleContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    alignItems: 'flex-start',
    flexWrap: 'wrap',
    gap: 8,
  },
  buttonWrapper: {
    alignItems: 'center',
    minWidth: 40,
  },
  button: {
    justifyContent: 'center',
    alignItems: 'center',
    borderWidth: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    elevation: 2,
  },
  buttonDisabled: {
    opacity: 0.5,
  },
  buttonText: {
    fontWeight: '600',
  },
  label: {
    marginTop: 4,
    fontSize: 10,
    color: COLORS.textSecondary,
    textAlign: 'center',
  },
});

export default RatingScale;
