import React from 'react';
import { View, Text, StyleSheet, TouchableOpacity } from 'react-native';
import { MoodResponse } from '../types';
import { COLORS, getMoodColor } from '../utils/constants';
import { formatDateTime, getRatingLabel } from '../utils/helpers';

interface MoodCardProps {
  response: MoodResponse;
  onPress?: () => void;
  showDate?: boolean;
}

export const MoodCard: React.FC<MoodCardProps> = ({
  response,
  onPress,
  showDate = true,
}) => {
  const moodColor = getMoodColor(response.rating, response.scaleMax);
  const label = getRatingLabel(response.rating, response.scaleMax);

  return (
    <TouchableOpacity
      style={styles.container}
      onPress={onPress}
      disabled={!onPress}
      activeOpacity={onPress ? 0.7 : 1}
    >
      <View style={[styles.indicator, { backgroundColor: moodColor }]} />
      <View style={styles.content}>
        <Text style={styles.question} numberOfLines={2}>
          {response.question}
        </Text>
        <View style={styles.ratingRow}>
          <View style={[styles.ratingBadge, { backgroundColor: moodColor }]}>
            <Text style={styles.ratingText}>
              {response.rating}/{response.scaleMax}
            </Text>
          </View>
          <Text style={[styles.label, { color: moodColor }]}>{label}</Text>
        </View>
        {showDate && (
          <Text style={styles.timestamp}>
            {formatDateTime(response.timestamp)}
          </Text>
        )}
        {response.context && (
          <Text style={styles.context} numberOfLines={2}>
            {response.context}
          </Text>
        )}
      </View>
    </TouchableOpacity>
  );
};

const styles = StyleSheet.create({
  container: {
    flexDirection: 'row',
    backgroundColor: COLORS.surface,
    borderRadius: 12,
    marginVertical: 6,
    marginHorizontal: 16,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
    overflow: 'hidden',
  },
  indicator: {
    width: 4,
  },
  content: {
    flex: 1,
    padding: 12,
  },
  question: {
    fontSize: 15,
    fontWeight: '500',
    color: COLORS.text,
    marginBottom: 8,
  },
  ratingRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 4,
  },
  ratingBadge: {
    paddingHorizontal: 10,
    paddingVertical: 4,
    borderRadius: 12,
    marginRight: 8,
  },
  ratingText: {
    fontSize: 14,
    fontWeight: '700',
    color: COLORS.surface,
  },
  label: {
    fontSize: 13,
    fontWeight: '500',
  },
  timestamp: {
    fontSize: 12,
    color: COLORS.textSecondary,
    marginTop: 4,
  },
  context: {
    fontSize: 13,
    color: COLORS.textSecondary,
    marginTop: 8,
    fontStyle: 'italic',
  },
});

export default MoodCard;
