import React from 'react';
import { View, Text, TouchableOpacity, StyleSheet } from 'react-native';
import { MoodResponse } from '../types';
import { COLORS, getMoodColor } from '../utils/constants';

interface CalendarDayProps {
  day: number;
  date: Date;
  responses: MoodResponse[];
  isToday: boolean;
  isCurrentMonth: boolean;
  onPress: (date: Date, responses: MoodResponse[]) => void;
}

export const CalendarDay: React.FC<CalendarDayProps> = ({
  day,
  date,
  responses,
  isToday,
  isCurrentMonth,
  onPress,
}) => {
  const hasData = responses.length > 0;

  // Calculate average mood for the day
  const averageMood = hasData
    ? responses.reduce((sum, r) => sum + r.rating / r.scaleMax, 0) /
      responses.length
    : null;

  const getMoodIndicatorColor = (): string => {
    if (!hasData || averageMood === null) return 'transparent';

    // Normalize to 1-5 scale equivalent
    const normalizedRating = Math.round(averageMood * 5);
    return getMoodColor(normalizedRating, 5);
  };

  return (
    <TouchableOpacity
      style={[
        styles.container,
        isToday && styles.today,
        !isCurrentMonth && styles.otherMonth,
      ]}
      onPress={() => onPress(date, responses)}
      disabled={!isCurrentMonth}
      activeOpacity={0.7}
    >
      <Text
        style={[
          styles.dayText,
          isToday && styles.todayText,
          !isCurrentMonth && styles.otherMonthText,
        ]}
      >
        {day}
      </Text>
      {hasData && isCurrentMonth && (
        <View
          style={[
            styles.indicator,
            { backgroundColor: getMoodIndicatorColor() },
          ]}
        >
          <Text style={styles.responseCount}>{responses.length}</Text>
        </View>
      )}
    </TouchableOpacity>
  );
};

const styles = StyleSheet.create({
  container: {
    width: '14.28%', // 100% / 7 days
    aspectRatio: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 4,
  },
  today: {
    backgroundColor: COLORS.primary + '20',
    borderRadius: 8,
  },
  otherMonth: {
    opacity: 0.3,
  },
  dayText: {
    fontSize: 14,
    fontWeight: '500',
    color: COLORS.text,
  },
  todayText: {
    color: COLORS.primary,
    fontWeight: '700',
  },
  otherMonthText: {
    color: COLORS.textSecondary,
  },
  indicator: {
    position: 'absolute',
    bottom: 4,
    width: 18,
    height: 18,
    borderRadius: 9,
    justifyContent: 'center',
    alignItems: 'center',
  },
  responseCount: {
    fontSize: 10,
    fontWeight: '700',
    color: COLORS.surface,
  },
});

export default CalendarDay;
