#!/bin/bash

# Mood Tracker - macOS Setup Script
# Run this script in your terminal on macOS

set -e

echo "=============================================="
echo "  Mood Tracker - macOS Setup"
echo "=============================================="
echo ""

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Function to check if command exists
check_command() {
    if command -v "$1" &> /dev/null; then
        echo -e "${GREEN}✓${NC} $1 is installed"
        return 0
    else
        echo -e "${RED}✗${NC} $1 is NOT installed"
        return 1
    fi
}

# Function to print step
print_step() {
    echo ""
    echo -e "${YELLOW}▶ $1${NC}"
    echo "----------------------------------------------"
}

#################################################
# STEP 1: Check Prerequisites
#################################################
print_step "Step 1: Checking Prerequisites"

MISSING_DEPS=0

# Check Node.js
if check_command node; then
    NODE_VERSION=$(node --version)
    echo "  Version: $NODE_VERSION"
else
    echo "  Install from: https://nodejs.org/"
    MISSING_DEPS=1
fi

# Check npm
if check_command npm; then
    NPM_VERSION=$(npm --version)
    echo "  Version: $NPM_VERSION"
else
    MISSING_DEPS=1
fi

# Check Watchman (optional but recommended)
if check_command watchman; then
    WATCHMAN_VERSION=$(watchman --version 2>/dev/null || echo "unknown")
    echo "  Version: $WATCHMAN_VERSION"
else
    echo "  (Optional) Install with: brew install watchman"
fi

# Check Xcode
if xcode-select -p &> /dev/null; then
    echo -e "${GREEN}✓${NC} Xcode Command Line Tools installed"
    XCODE_PATH=$(xcode-select -p)
    echo "  Path: $XCODE_PATH"
else
    echo -e "${RED}✗${NC} Xcode Command Line Tools NOT installed"
    echo "  Install with: xcode-select --install"
    MISSING_DEPS=1
fi

# Check Xcode version
if command -v xcodebuild &> /dev/null; then
    XCODE_VERSION=$(xcodebuild -version 2>/dev/null | head -1 || echo "unknown")
    echo "  $XCODE_VERSION"
else
    echo -e "${RED}✗${NC} Xcode is NOT installed"
    echo "  Install from: App Store"
    MISSING_DEPS=1
fi

# Check CocoaPods
if check_command pod; then
    POD_VERSION=$(pod --version)
    echo "  Version: $POD_VERSION"
else
    echo "  Install with: sudo gem install cocoapods"
    MISSING_DEPS=1
fi

# Check Ruby (for CocoaPods)
if check_command ruby; then
    RUBY_VERSION=$(ruby --version | head -1)
    echo "  Version: $RUBY_VERSION"
fi

if [ $MISSING_DEPS -eq 1 ]; then
    echo ""
    echo -e "${RED}Some required dependencies are missing.${NC}"
    echo "Please install them and run this script again."
    echo ""
    read -p "Do you want to continue anyway? (y/N): " CONTINUE
    if [[ ! $CONTINUE =~ ^[Yy]$ ]]; then
        exit 1
    fi
fi

#################################################
# STEP 2: Navigate to Project Directory
#################################################
print_step "Step 2: Setting Up Project Directory"

# Check if we're in the right directory
if [ -f "package.json" ] && grep -q "MoodTracker" package.json 2>/dev/null; then
    echo -e "${GREEN}✓${NC} Already in MoodTracker project directory"
    PROJECT_DIR=$(pwd)
else
    # Check if MoodTracker exists in current directory
    if [ -d "MoodTracker" ]; then
        cd MoodTracker
        PROJECT_DIR=$(pwd)
        echo -e "${GREEN}✓${NC} Found MoodTracker directory"
    elif [ -d "symptoms" ]; then
        cd symptoms
        PROJECT_DIR=$(pwd)
        echo -e "${GREEN}✓${NC} Found symptoms directory"
    else
        echo "Current directory: $(pwd)"
        echo ""
        echo "Please run this script from the MoodTracker project directory,"
        echo "or copy the project files to this location first."
        echo ""
        read -p "Enter the path to the MoodTracker project (or press Enter to use current directory): " PROJECT_PATH
        if [ -n "$PROJECT_PATH" ]; then
            cd "$PROJECT_PATH"
        fi
        PROJECT_DIR=$(pwd)
    fi
fi

echo "Project directory: $PROJECT_DIR"

#################################################
# STEP 3: Install Node Dependencies
#################################################
print_step "Step 3: Installing Node.js Dependencies"

if [ -f "package.json" ]; then
    echo "Running npm install..."
    npm install
    echo -e "${GREEN}✓${NC} Node dependencies installed"
else
    echo -e "${RED}✗${NC} package.json not found!"
    echo "Make sure you're in the correct project directory."
    exit 1
fi

#################################################
# STEP 4: Install iOS Dependencies (CocoaPods)
#################################################
print_step "Step 4: Installing iOS Dependencies (CocoaPods)"

if [ -d "ios" ]; then
    cd ios

    # Check if Podfile exists
    if [ -f "Podfile" ]; then
        echo "Running pod install..."
        pod install
        echo -e "${GREEN}✓${NC} CocoaPods dependencies installed"
    else
        echo -e "${YELLOW}!${NC} Podfile not found, skipping pod install"
    fi

    cd ..
else
    echo -e "${YELLOW}!${NC} ios directory not found, skipping iOS setup"
fi

#################################################
# STEP 5: Verify Android Setup (Optional)
#################################################
print_step "Step 5: Checking Android Setup (Optional)"

if [ -n "$ANDROID_HOME" ]; then
    echo -e "${GREEN}✓${NC} ANDROID_HOME is set: $ANDROID_HOME"
elif [ -n "$ANDROID_SDK_ROOT" ]; then
    echo -e "${GREEN}✓${NC} ANDROID_SDK_ROOT is set: $ANDROID_SDK_ROOT"
else
    echo -e "${YELLOW}!${NC} Android SDK not configured"
    echo "  If you want to run on Android, set ANDROID_HOME or ANDROID_SDK_ROOT"
    echo "  Default location: ~/Library/Android/sdk"
fi

#################################################
# STEP 6: Setup Complete
#################################################
print_step "Setup Complete!"

echo ""
echo -e "${GREEN}=============================================="
echo "  Mood Tracker is ready!"
echo "==============================================${NC}"
echo ""
echo "To run the app:"
echo ""
echo "  1. Start Metro bundler (in one terminal):"
echo "     ${YELLOW}npm start${NC}"
echo ""
echo "  2. Run on iOS (in another terminal):"
echo "     ${YELLOW}npm run ios${NC}"
echo ""
echo "  3. Or run on Android:"
echo "     ${YELLOW}npm run android${NC}"
echo ""
echo "Useful commands:"
echo "  • npm run ios              - Run on iOS simulator"
echo "  • npm run android          - Run on Android emulator"
echo "  • npm start -- --reset-cache  - Clear Metro cache"
echo "  • npm run typecheck        - Check TypeScript"
echo ""
echo "Project location: $PROJECT_DIR"
echo ""

# Ask if user wants to start the app
read -p "Do you want to start the Metro bundler now? (y/N): " START_METRO
if [[ $START_METRO =~ ^[Yy]$ ]]; then
    echo ""
    echo "Starting Metro bundler..."
    echo "Press Ctrl+C to stop, then run 'npm run ios' in another terminal"
    echo ""
    npm start
fi
