# Mood Tracker App - Setup & Deployment Guide

## Overview

Mood Tracker is a cross-platform mobile app (iOS & Android) that tracks user mood and feelings throughout the day using background notifications with actionable rating buttons.

## Prerequisites

### Required Software

1. **Node.js** (v18 or later)
   ```bash
   node --version  # Should be 18.x or higher
   ```

2. **npm** (v9 or later)
   ```bash
   npm --version
   ```

3. **Watchman** (for file watching)
   ```bash
   # macOS
   brew install watchman
   ```

4. **For iOS Development:**
   - macOS (required)
   - Xcode 15 or later
   - CocoaPods
   ```bash
   sudo gem install cocoapods
   ```

5. **For Android Development:**
   - Android Studio
   - Android SDK (API level 33 recommended)
   - JDK 17

### Environment Setup

Follow the official React Native environment setup guide:
https://reactnative.dev/docs/environment-setup

Select "React Native CLI Quickstart" (not Expo).

## Project Installation

### 1. Install Dependencies

```bash
cd /opt/apps/construction/pvt/symptoms
npm install
```

### 2. iOS Setup

```bash
# Navigate to iOS directory
cd ios

# Install CocoaPods dependencies
pod install

# Return to project root
cd ..
```

### 3. Android Setup

The Android project should be ready after `npm install`. If you need to sync Gradle:

```bash
cd android
./gradlew clean
cd ..
```

## Running the App

### iOS Simulator

```bash
# Start Metro bundler (in separate terminal)
npm start

# Run on iOS simulator
npx react-native run-ios

# Or specify a device
npx react-native run-ios --simulator="iPhone 15"
```

### Android Emulator

```bash
# Start Metro bundler (in separate terminal)
npm start

# Ensure Android emulator is running or device is connected
# List devices:
adb devices

# Run on Android
npx react-native run-android
```

### Running on Physical Devices

#### iOS Device

1. Open `ios/MoodTracker.xcworkspace` in Xcode
2. Select your connected device
3. Configure signing (requires Apple Developer account):
   - Select the MoodTracker target
   - Go to "Signing & Capabilities"
   - Select your team
4. Click the Run button (Cmd+R)

#### Android Device

1. Enable USB debugging on your device
2. Connect via USB
3. Run:
   ```bash
   npx react-native run-android
   ```

## Permission Handling

### iOS Permissions

The app requests the following permissions:
- **Notifications** - Required for mood check-in prompts
- **Background App Refresh** - Required for scheduling notifications when app is closed

Users must grant these permissions during onboarding.

### Android Permissions

The following permissions are configured in `AndroidManifest.xml`:
- `POST_NOTIFICATIONS` - For sending notifications (Android 13+)
- `RECEIVE_BOOT_COMPLETED` - To reschedule notifications after device restart
- `SCHEDULE_EXACT_ALARM` - For precise notification timing
- `WAKE_LOCK` - To wake device for notifications
- `VIBRATE` - For notification vibration

#### Battery Optimization (Android)

For reliable background notifications, users should disable battery optimization for the app:
1. Settings > Apps > Mood Tracker > Battery
2. Select "Unrestricted" or "Don't optimize"

## App Features

### 1. Background Notifications
- Notifications appear at scheduled times with rating buttons
- Users can tap 1-5 (or 1-10) directly from the notification
- No need to open the app to respond

### 2. Flexible Scheduling
- Set multiple check-in times per day
- Configure which days of the week
- Add random time variance (+/- minutes)

### 3. Customizable Questions
- Default questions included (mood, energy, stress, etc.)
- Add custom questions
- Enable/disable individual questions

### 4. Rating Scales
- 1-5 scale (default)
- 1-10 scale
- 0-10 scale

### 5. Data Visualization
- Dashboard with weekly trends
- History list with filtering
- Calendar view with mood indicators
- Victory Native charts

### 6. Local Data Storage
- All data stored locally using SQLite
- No account required
- Privacy-focused design

### 7. Optional Cloud Sync
- Framework ready for Firebase integration
- See `src/services/cloudSync.ts` for implementation details

## Project Structure

```
symptoms/
├── src/
│   ├── components/          # Reusable UI components
│   │   ├── RatingScale.tsx
│   │   ├── MoodCard.tsx
│   │   └── CalendarDay.tsx
│   ├── screens/             # App screens
│   │   ├── HomeScreen.tsx
│   │   ├── HistoryScreen.tsx
│   │   ├── CalendarScreen.tsx
│   │   ├── SettingsScreen.tsx
│   │   └── OnboardingScreen.tsx
│   ├── services/            # Business logic
│   │   ├── database.ts      # SQLite operations
│   │   ├── notifications.ts # Notifee handling
│   │   ├── scheduler.ts     # Notification scheduling
│   │   ├── background.ts    # Background tasks
│   │   └── cloudSync.ts     # Optional cloud sync
│   ├── stores/              # State management
│   │   └── appStore.ts      # Zustand store
│   ├── hooks/               # Custom React hooks
│   ├── types/               # TypeScript types
│   ├── utils/               # Utilities & constants
│   └── navigation/          # Navigation setup
├── android/                 # Android native code
├── ios/                     # iOS native code
├── App.tsx                  # Root component
└── package.json
```

## Building for Production

### iOS Build

1. Open `ios/MoodTracker.xcworkspace` in Xcode
2. Select "Generic iOS Device" as the build target
3. Product > Archive
4. Once archived, click "Distribute App"
5. Choose your distribution method (App Store, Ad Hoc, etc.)

### Android Build

#### Debug APK

```bash
cd android
./gradlew assembleDebug
# APK location: android/app/build/outputs/apk/debug/app-debug.apk
```

#### Release APK

1. Generate a signing key:
   ```bash
   keytool -genkeypair -v -storetype PKCS12 -keystore moodtracker.keystore -alias moodtracker -keyalg RSA -keysize 2048 -validity 10000
   ```

2. Add signing config to `android/app/build.gradle`:
   ```gradle
   signingConfigs {
       release {
           storeFile file('moodtracker.keystore')
           storePassword 'your-password'
           keyAlias 'moodtracker'
           keyPassword 'your-password'
       }
   }
   buildTypes {
       release {
           signingConfig signingConfigs.release
       }
   }
   ```

3. Build release:
   ```bash
   cd android
   ./gradlew assembleRelease
   # APK location: android/app/build/outputs/apk/release/app-release.apk
   ```

#### Android App Bundle (AAB) for Play Store

```bash
cd android
./gradlew bundleRelease
# AAB location: android/app/build/outputs/bundle/release/app-release.aab
```

## Troubleshooting

### Common Issues

1. **Metro bundler issues**
   ```bash
   npm start -- --reset-cache
   ```

2. **iOS build errors**
   ```bash
   cd ios
   pod deintegrate
   pod install
   cd ..
   npx react-native run-ios
   ```

3. **Android build errors**
   ```bash
   cd android
   ./gradlew clean
   cd ..
   npx react-native run-android
   ```

4. **Notifications not appearing (Android)**
   - Check battery optimization settings
   - Ensure POST_NOTIFICATIONS permission is granted (Android 13+)
   - Check notification channel settings

5. **Background tasks not running (iOS)**
   - Verify Background App Refresh is enabled in device settings
   - Check Info.plist has UIBackgroundModes configured

### Debug Tools

- **React Native Debugger**: Download from https://github.com/jhen0409/react-native-debugger
- **Flipper**: Download from https://fbflipper.com/

## Testing Notifications

To test notifications during development:

1. Open the app and complete onboarding
2. Go to Settings > tap "Refresh Notifications"
3. Or use the "Quick Check-In Now" button on the Home screen

To test background notification handling:
1. Schedule a notification for a time in the near future
2. Close the app completely
3. Wait for the notification to appear
4. Tap a rating button to verify it's recorded

## Cloud Sync Integration (Optional)

To enable Firebase cloud sync:

1. Install Firebase dependencies:
   ```bash
   npm install @react-native-firebase/app @react-native-firebase/firestore @react-native-firebase/auth
   ```

2. Follow Firebase setup for iOS:
   - Add `GoogleService-Info.plist` to iOS project
   - Update `AppDelegate.mm`

3. Follow Firebase setup for Android:
   - Add `google-services.json` to `android/app/`
   - Update `build.gradle` files

4. Update `src/services/cloudSync.ts` with actual Firebase implementation

See the Firebase React Native documentation:
https://rnfirebase.io/

## License

MIT License - Feel free to modify and distribute.
